# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # Feedback: (drawer-open drawer_bottom)
        # The goal is to discover which predicate is missing, likely related to the state of 'drawer_bottom'.
        # We will attempt to interact with the drawer and observe the effects.

        # For this example, we assume the following object names and locations:
        # - drawer_bottom: the drawer to be explored
        # - handle_bottom: the handle object for the drawer
        # - switch: the light switch object
        # - robot initial location: 'init_pos'
        # - drawer location: 'drawer_pos'
        # - switch location: 'switch_pos'
        # These names should match those in your environment and object_positions.

        # You may need to adjust these names based on your environment.
        try:
            drawer_name = 'drawer_bottom'
            handle_name = 'handle_bottom'
            switch_name = 'switch'
            # Get locations from positions dict
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            switch_pos = positions.get(switch_name, None)
            # For simplicity, assume robot starts at 'init_pos'
            robot_pos = positions.get('init_pos', None)
        except Exception as e:
            print("[Error] Could not retrieve object positions:", e)
            return

        # Step 1: If the room is dark, turn on the light
        # We assume the environment provides a way to check room brightness in obs or descriptions
        room_is_dark = False
        if 'room-dark' in descriptions or 'room-dark' in obs.get('state', []):
            room_is_dark = True
        if room_is_dark:
            try:
                print("[Exploration] Room is dark, turning on the light.")
                obs, reward, done = execute_push_switch(env, task, switch_name, switch_pos)
            except Exception as e:
                print("[Error] Failed to execute_push_switch:", e)
                return

        # Step 2: Move to the handle location if not already there
        try:
            current_pos = robot_pos
            if current_pos != handle_pos:
                print(f"[Exploration] Moving robot from {current_pos} to handle at {handle_pos}.")
                obs, reward, done = execute_go(env, task, current_pos, handle_pos)
                current_pos = handle_pos
        except Exception as e:
            print("[Error] Failed to execute_go to handle:", e)
            return

        # Step 3: Pick the handle (if not already holding)
        try:
            print(f"[Exploration] Attempting to pick handle: {handle_name} at {handle_pos}.")
            obs, reward, done = execute_pick(env, task, handle_name, handle_pos)
        except Exception as e:
            print("[Error] Failed to execute_pick handle:", e)
            return

        # Step 4: Attempt to pull the drawer open
        # This may fail if the drawer is locked or preconditions are not met
        try:
            print(f"[Exploration] Attempting to pull open drawer: {drawer_name} using handle: {handle_name}.")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, handle_pos)
            # If successful, the predicate (drawer-open drawer_bottom) should now be true
            print("[Exploration] Drawer pull executed. Check if drawer is open in observation.")
        except Exception as e:
            print("[Error] Failed to execute_pull drawer:", e)
            # If this fails, the missing predicate may be related to lock state or other preconditions
            # Exploration: Try to discover lock state if possible
            print("[Exploration] Drawer may be locked or missing predicate. Exploration needed.")
            # If available, try to use execute_pull again to discover lock-known predicate
            # (Not available in primitive_skills_static_lock_v2, so we stop here)
            return

        # Step 5: Check if the drawer is open (predicate discovered)
        # In a real system, you would check the updated state/predicates in obs or descriptions
        # For demonstration, print the feedback and conclude exploration
        print("[Exploration] If drawer is now open, predicate (drawer-open drawer_bottom) is true.")
        print("[Exploration] Exploration phase complete. Missing predicate likely related to drawer state (e.g., drawer-locked, drawer-unlocked, drawer-open).")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
