# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan is timing out, likely due to a missing predicate in the domain/problem.
        # We will use the exploration actions to probe the environment and determine which predicate is missing.
        # This phase is crucial for debugging and for the planner to succeed.

        print("[Exploration] Starting exploration to identify missing predicates...")

        # Example: Try to identify all objects at all locations
        # (Assume positions is a dict: {object_name: (x, y, z, location_name)})
        # We'll attempt to use available skills to probe for missing knowledge.

        # For demonstration, we will try to use the available skills in a safe way.
        # Since we do not know the exact objects/locations, we will iterate over what we have.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # Try to use exploration actions to discover missing predicates
        # For each object, try to pick, pull, or otherwise interact to see if any predicate is missing
        for obj_name, obj_info in positions.items():
            # obj_info may be (x, y, z, location_name) or similar
            if isinstance(obj_info, dict):
                location = obj_info.get('location', None)
            elif isinstance(obj_info, (tuple, list)) and len(obj_info) >= 4:
                location = obj_info[3]
            else:
                location = None

            if location is None:
                continue

            print(f"[Exploration] Probing object '{obj_name}' at location '{location}'")

            # Try to pick the object (if possible)
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name,
                    location
                )
                print(f"[Exploration] execute_pick succeeded for {obj_name} at {location}")
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj_name} at {location}: {e}")

            # Try to pull the object (if possible)
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    obj_name,
                    location
                )
                print(f"[Exploration] execute_pull succeeded for {obj_name} at {location}")
            except Exception as e:
                print(f"[Exploration] execute_pull failed for {obj_name} at {location}: {e}")

            # Try to sweep the object (if possible)
            try:
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name,
                    location
                )
                print(f"[Exploration] execute_sweep succeeded for {obj_name} at {location}")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj_name} at {location}: {e}")

        # Try to move the robot between locations (if possible)
        # Collect all unique locations
        all_locations = set()
        for obj_info in positions.values():
            if isinstance(obj_info, dict):
                loc = obj_info.get('location', None)
            elif isinstance(obj_info, (tuple, list)) and len(obj_info) >= 4:
                loc = obj_info[3]
            else:
                loc = None
            if loc is not None:
                all_locations.add(loc)
        all_locations = list(all_locations)

        if len(all_locations) >= 2:
            from_loc = all_locations[0]
            to_loc = all_locations[1]
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                print(f"[Exploration] execute_go succeeded from {from_loc} to {to_loc}")
            except Exception as e:
                print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # Try to push a switch if available (for light)
        # The skill name is 'execute_push_switch' but in the available skills list it's written as 'execute_push_swtich -> push'
        # We'll try both names for robustness
        for switch_skill in ['execute_push_switch', 'execute_push_swtich']:
            try:
                obs, reward, done = globals()[switch_skill](
                    env,
                    task,
                    # Assume we can get a switch and location from positions or task
                    # For now, use the first object as a switch if possible
                    list(positions.keys())[0],
                    all_locations[0] if all_locations else None
                )
                print(f"[Exploration] {switch_skill} succeeded")
                break
            except Exception as e:
                print(f"[Exploration] {switch_skill} failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan.
        # For this skeleton, you would insert the plan steps here, using only the available skills.
        # Example (commented out, as actual plan is not provided):
        #
        # obs, reward, done = execute_pick(env, task, 'object_1', 'location_1')
        # obs, reward, done = execute_go(env, task, 'location_1', 'location_2')
        # obs, reward, done = execute_place(env, task, 'object_1', 'drawer_1', 'location_2')
        #
        # if done:
        #     print("[Task] Task completed successfully!")
        #     return

        # TODO: Insert actual oracle plan steps here using only the provided skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
