# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (drawer-unlocked drawer_bottom) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer_bottom) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We use the available skills to interact with the drawer and infer its state.

        # Assumptions for object names (update as per your environment):
        # - The drawer is named 'drawer_bottom'
        # - The handle object for the drawer is 'handle_bottom'
        # - The robot starts at a location, e.g., 'location_a'
        # - The drawer is at 'location_drawer'
        # - The switch is 'switch_main' at 'location_switch'
        # - The object to pick is 'target_object' at 'location_object'
        # These names should be replaced with actual names from your environment if different.

        # Example object/location names (replace with actual names if needed)
        drawer_name = 'drawer_bottom'
        handle_name = 'handle_bottom'
        switch_name = 'switch_main'
        object_name = 'target_object'
        robot_location = 'location_a'
        drawer_location = 'location_drawer'
        switch_location = 'location_switch'
        object_location = 'location_object'

        # Try to get actual positions/names from the environment if available
        # (If your get_object_positions() returns more detailed info, use it here)
        try:
            if 'drawer_bottom' in positions:
                drawer_location = positions['drawer_bottom']
            if 'handle_bottom' in positions:
                handle_location = positions['handle_bottom']
            if 'switch_main' in positions:
                switch_location = positions['switch_main']
            if 'target_object' in positions:
                object_location = positions['target_object']
        except Exception as e:
            print("[Warning] Could not extract all object positions:", e)

        # === Step 1: Ensure Room is Bright (if not, push the switch) ===
        # Try to turn on the light if the room is dark
        try:
            print("[Exploration] Ensuring room is bright by pushing the switch if needed.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
        except Exception as e:
            print("[Info] execute_push_switch may not be needed or already executed:", e)

        # === Step 2: Try to Pull the Drawer to Check Lock State ===
        # The feedback suggests we need to check if the drawer is unlocked.
        # We attempt to pull the drawer using the handle.
        # If the drawer is locked, the action may fail or provide feedback.

        # First, pick up the handle if required
        try:
            print("[Exploration] Picking up the handle to attempt pulling the drawer.")
            obs, reward, done = execute_pick(
                env,
                task,
                handle_name,
                drawer_location  # Assume handle is at the drawer location
            )
        except Exception as e:
            print("[Warning] Could not pick up handle:", e)

        # Now, try to pull the drawer
        try:
            print("[Exploration] Attempting to pull the drawer to check if it is unlocked.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                drawer_location
            )
            print("[Exploration] Drawer pull attempted. If successful, drawer is unlocked.")
            # If this succeeds, we can infer (drawer-unlocked drawer_bottom)
        except Exception as e:
            print("[Exploration] Drawer pull failed. Drawer may be locked or another issue:", e)
            # If this fails, the drawer may be locked, or another precondition is missing

        # === Step 3: (Optional) Continue with Task Plan if Drawer is Unlocked ===
        # If the drawer is now open, proceed with the rest of the plan (e.g., pick and place object)
        # For demonstration, we attempt to pick an object and place it in the drawer

        # Pick up the target object (if required)
        try:
            print("[Task] Picking up the target object.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name,
                object_location
            )
        except Exception as e:
            print("[Warning] Could not pick up target object:", e)

        # Place the object in the drawer (if drawer is open)
        try:
            print("[Task] Placing the object in the drawer.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name,
                drawer_name,
                drawer_location
            )
        except Exception as e:
            print("[Warning] Could not place object in drawer:", e)

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
