# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the robot's location and update our knowledge.

        # For demonstration, let's assume the robot starts at 'init_pos' and we want to check if 'floor' is a valid location.
        # We'll use the available skills to move the robot and observe the result.

        # Find all locations from positions
        location_names = [name for name, pos in positions.items() if 'floor' in name or 'room' in name or 'drawer' in name or 'table' in name or 'loc' in name]
        if not location_names:
            # Fallback: try to find any location-like keys
            location_names = [name for name in positions.keys() if 'loc' in name or 'floor' in name or 'room' in name]

        # Try to find the robot's current location
        robot_location = None
        for name in positions.keys():
            if 'robot' in name:
                robot_location = name
                break
        if robot_location is None:
            # Fallback: assume starting at the first location
            robot_location = location_names[0] if location_names else None

        # Exploration: Try to move the robot to 'floor' if possible
        floor_location = None
        for loc in location_names:
            if 'floor' in loc:
                floor_location = loc
                break

        # If 'floor' is a valid location, try to move there using execute_go
        if floor_location and robot_location and floor_location != robot_location:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {floor_location} to check predicate (robot-at floor)")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[robot_location],
                    to_location=positions[floor_location],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {floor_location}.")
            except Exception as e:
                print(f"[Exploration] Failed to move robot to {floor_location}: {e}")
        else:
            print("[Exploration] Could not find a valid 'floor' location or robot location for exploration.")

        # After exploration, you would update your internal state or knowledge base if needed.
        # For this skeleton, we simply demonstrate the exploration step.

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan step-by-step using only the provided skills.
        # For example:
        # 1. Turn on the light if the room is dark
        # 2. Pick up an object from the floor
        # 3. Move to a drawer and open it
        # 4. Place the object in the drawer
        # (The actual plan steps depend on your specific task and initial state.)

        # Example (commented, as actual object names and plan are not provided):
        # if 'switch' in positions and 'floor' in positions:
        #     obs, reward, done = execute_push_switch(
        #         env,
        #         task,
        #         switch_name='switch',
        #         location=positions['floor'],
        #         approach_distance=0.15,
        #         max_steps=100,
        #         threshold=0.01,
        #         approach_axis='z',
        #         timeout=10.0
        #     )
        #     print("[Task] Light switched on.")

        # Continue with pick, place, pull, push, etc., as per the oracle plan and available skills.

        # TODO: Insert the actual oracle plan steps here, using only the provided skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
