# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and confirm the robot's location predicates.

        # Try to use execute_go to move the robot and observe the effect
        # Since we don't know the exact names of locations, try to infer from positions
        location_names = []
        for name, pos in positions.items():
            if "floor" in name or "room" in name or "location" in name:
                location_names.append(name)
        # If no location found, just use all keys as fallback
        if not location_names:
            location_names = list(positions.keys())

        # For exploration, try to move the robot between two locations if possible
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            print(f"[Exploration] Attempting to move robot from {from_loc} to {to_loc} using execute_go.")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[from_loc],
                    to_location=positions[to_loc],
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to move robot from {from_loc} to {to_loc}: {e}")
        else:
            print("[Exploration] Not enough locations found to perform execute_go.")

        # After exploration, print out the current robot location predicate
        # (Assume task.get_observation() returns a dict with predicates or state info)
        try:
            current_obs = task.get_observation()
            if isinstance(current_obs, dict):
                robot_at = None
                for k, v in current_obs.items():
                    if "robot-at" in k or "robot_at" in k:
                        robot_at = (k, v)
                        break
                if robot_at:
                    print(f"[Exploration] Current robot location predicate: {robot_at}")
                else:
                    print("[Exploration] Could not find robot location predicate in observation.")
            else:
                print("[Exploration] Observation is not a dict; cannot parse predicates.")
        except Exception as e:
            print(f"[Exploration] Error retrieving observation: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan using available skills as needed.
        # For this exploration task, we stop after identifying the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()