# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available exploration skills to discover missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will try to use the available skills to trigger exploration actions.

        # Since we do not have the actual oracle plan, we perform a systematic exploration using available skills.
        # We assume the environment provides a list of objects, drawers, locations, and switches.
        # We'll attempt to use each skill in a safe, exploratory way to see if any missing predicate is revealed.

        # Get lists of objects, drawers, locations, switches from positions or descriptions if available
        objects = []
        drawers = []
        locations = []
        switches = []
        handles = []

        # Try to extract object types from positions or descriptions
        if hasattr(task, 'get_objects_by_type'):
            objects = task.get_objects_by_type('object')
            drawers = task.get_objects_by_type('drawer')
            locations = task.get_objects_by_type('location')
            switches = task.get_objects_by_type('switch')
            handles = task.get_objects_by_type('handle')
        else:
            # Fallback: try to infer from positions dict
            for name in positions:
                if 'drawer' in name:
                    drawers.append(name)
                elif 'switch' in name:
                    switches.append(name)
                elif 'handle' in name:
                    handles.append(name)
                elif 'loc' in name or 'room' in name:
                    locations.append(name)
                else:
                    objects.append(name)

        # Remove duplicates
        objects = list(set(objects))
        drawers = list(set(drawers))
        locations = list(set(locations))
        switches = list(set(switches))
        handles = list(set(handles))

        # If no locations found, try to get robot's current location from observation
        robot_location = None
        if hasattr(task, 'get_robot_location'):
            robot_location = task.get_robot_location()
            if robot_location and robot_location not in locations:
                locations.append(robot_location)
        elif 'robot' in positions:
            robot_location = positions['robot']
            if robot_location and robot_location not in locations:
                locations.append(robot_location)
        elif len(locations) == 0:
            # Fallback: use a default location name
            locations = ['locA', 'locB']

        # Print for debug
        print("[Exploration] Objects:", objects)
        print("[Exploration] Drawers:", drawers)
        print("[Exploration] Locations:", locations)
        print("[Exploration] Switches:", switches)
        print("[Exploration] Handles:", handles)

        # Try to execute each available skill in a safe way to trigger missing predicates
        # Only use skills that are available in skill_code and in the provided skill list

        # 1. Try to turn on the light if possible (execute_push_switch)
        try:
            if 'execute_push_switch' in globals() or 'execute_push_switch' in locals():
                for s in switches:
                    for p in locations:
                        print(f"[Exploration] Trying execute_push_switch on switch {s} at location {p}")
                        try:
                            obs, reward, done = execute_push_switch(env, task, s, p)
                            if done:
                                print("[Exploration] Light switched on, room should be bright now.")
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_push_switch failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_push_switch: {e}")

        # 2. Try to pick up each object (execute_pick)
        try:
            if 'execute_pick' in globals() or 'execute_pick' in locals():
                for o in objects:
                    for p in locations:
                        print(f"[Exploration] Trying execute_pick on object {o} at location {p}")
                        try:
                            obs, reward, done = execute_pick(env, task, o, p)
                            if done:
                                print(f"[Exploration] Picked up {o} at {p}")
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_pick failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_pick: {e}")

        # 3. Try to open each drawer (execute_pull)
        try:
            if 'execute_pull' in globals() or 'execute_pull' in locals():
                for d in drawers:
                    for h in handles:
                        for p in locations:
                            print(f"[Exploration] Trying execute_pull on drawer {d} with handle {h} at location {p}")
                            try:
                                obs, reward, done = execute_pull(env, task, d, h, p)
                                if done:
                                    print(f"[Exploration] Pulled drawer {d} with handle {h} at {p}")
                                    break
                            except Exception as e:
                                print(f"[Exploration] execute_pull failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_pull: {e}")

        # 4. Try to place each object in each drawer (execute_place)
        try:
            if 'execute_place' in globals() or 'execute_place' in locals():
                for o in objects:
                    for d in drawers:
                        for p in locations:
                            print(f"[Exploration] Trying execute_place for object {o} in drawer {d} at location {p}")
                            try:
                                obs, reward, done = execute_place(env, task, o, d, p)
                                if done:
                                    print(f"[Exploration] Placed {o} in {d} at {p}")
                                    break
                            except Exception as e:
                                print(f"[Exploration] execute_place failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_place: {e}")

        # 5. Try to push each drawer closed (execute_push)
        try:
            if 'execute_push' in globals() or 'execute_push' in locals():
                for d in drawers:
                    for p in locations:
                        print(f"[Exploration] Trying execute_push on drawer {d} at location {p}")
                        try:
                            obs, reward, done = execute_push(env, task, d, p)
                            if done:
                                print(f"[Exploration] Pushed drawer {d} at {p}")
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_push failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_push: {e}")

        # 6. Try to move between locations (execute_go)
        try:
            if 'execute_go' in globals() or 'execute_go' in locals():
                for i in range(len(locations)):
                    for j in range(len(locations)):
                        if i != j:
                            from_loc = locations[i]
                            to_loc = locations[j]
                            print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                            try:
                                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                                if done:
                                    print(f"[Exploration] Moved from {from_loc} to {to_loc}")
                                    break
                            except Exception as e:
                                print(f"[Exploration] execute_go failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_go: {e}")

        # 7. Try to sweep each object (execute_sweep)
        try:
            if 'execute_sweep' in globals() or 'execute_sweep' in locals():
                for o in objects:
                    for p in locations:
                        print(f"[Exploration] Trying execute_sweep on object {o} at location {p}")
                        try:
                            obs, reward, done = execute_sweep(env, task, o, p)
                            if done:
                                print(f"[Exploration] Swept {o} at {p}")
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_sweep failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_sweep: {e}")

        # 8. Try to use gripper (execute_gripper)
        try:
            if 'execute_gripper' in globals() or 'execute_gripper' in locals():
                print(f"[Exploration] Trying execute_gripper")
                try:
                    obs, reward, done = execute_gripper(env, task)
                    if done:
                        print(f"[Exploration] Used gripper")
                except Exception as e:
                    print(f"[Exploration] execute_gripper failed: {e}")
        except Exception as e:
            print(f"[Exploration] Error in execute_gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for any missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze feedback/logs to determine which predicate is missing.
        # If the missing predicate is discovered, update the plan or inform the user.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()