# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills to see which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # We assume positions contains keys for objects, drawers, locations, switches, etc.
        # We'll try to call each skill with plausible arguments and catch exceptions.

        # Example: Try to pick up each object on the floor at its location
        for obj_name, obj_info in positions.items():
            # Try execute_pick if the object is on the floor
            try:
                if obj_info.get('on_floor', False):
                    location = obj_info.get('location', None)
                    if location is not None:
                        print(f"[Exploration] Trying execute_pick on {obj_name} at {location}")
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            obj_name,
                            location
                        )
                        if done:
                            print(f"[Exploration] Successfully picked {obj_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj_name}: {e}")

        # Try to open each drawer using execute_pull if possible
        for drawer_name, drawer_info in positions.items():
            try:
                if drawer_info.get('type', '') == 'drawer':
                    handle = drawer_info.get('handle', None)
                    location = drawer_info.get('location', None)
                    if handle is not None and location is not None:
                        print(f"[Exploration] Trying execute_pull on {drawer_name} with handle {handle} at {location}")
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            drawer_name,
                            handle,
                            location
                        )
                        if done:
                            print(f"[Exploration] Successfully pulled {drawer_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_pull failed for {drawer_name}: {e}")

        # Try to push each drawer closed using execute_push
        for drawer_name, drawer_info in positions.items():
            try:
                if drawer_info.get('type', '') == 'drawer':
                    location = drawer_info.get('location', None)
                    if location is not None:
                        print(f"[Exploration] Trying execute_push on {drawer_name} at {location}")
                        obs, reward, done = execute_push(
                            env,
                            task,
                            drawer_name,
                            location
                        )
                        if done:
                            print(f"[Exploration] Successfully pushed {drawer_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_push failed for {drawer_name}: {e}")

        # Try to turn on the light using execute_push_switch if a switch is present
        for obj_name, obj_info in positions.items():
            try:
                if obj_info.get('type', '') == 'switch':
                    location = obj_info.get('location', None)
                    if location is not None:
                        print(f"[Exploration] Trying execute_push_switch on {obj_name} at {location}")
                        obs, reward, done = execute_push_switch(
                            env,
                            task,
                            obj_name,
                            location
                        )
                        if done:
                            print(f"[Exploration] Successfully pushed switch {obj_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_push_switch failed for {obj_name}: {e}")

        # Try to move the robot between locations using execute_go
        locations = [k for k, v in positions.items() if v.get('type', '') == 'location']
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # Try to sweep objects on the floor using execute_sweep
        for obj_name, obj_info in positions.items():
            try:
                if obj_info.get('on_floor', False):
                    location = obj_info.get('location', None)
                    if location is not None:
                        print(f"[Exploration] Trying execute_sweep on {obj_name} at {location}")
                        obs, reward, done = execute_sweep(
                            env,
                            task,
                            obj_name,
                            location
                        )
                        if done:
                            print(f"[Exploration] Successfully swept {obj_name}.")
                            break
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj_name}: {e}")

        # Try to use the gripper (if applicable)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()