# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan failed due to a missing predicate. We need to explore the environment
        # to discover which predicate is missing, using only the available skills.

        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the environment's response.

        # Get all objects and locations from the environment
        object_names = [name for name in positions.keys() if 'object' in name or 'obj' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'room' in name or 'drawer' in name]
        switch_names = [name for name in positions.keys() if 'switch' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        handle_names = [name for name in positions.keys() if 'handle' in name]

        # Fallback: If no objects/locations found, use some defaults
        if not object_names:
            object_names = ['object_1']
        if not location_names:
            location_names = ['location_1']
        if not switch_names:
            switch_names = ['switch_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not handle_names:
            handle_names = ['handle_1']

        # Try to turn on the light if the room is dark (execute_push_switch)
        try:
            print("[Exploration] Attempting to turn on the light (execute_push_switch)...")
            # Use the first switch and location as a guess
            switch = switch_names[0]
            location = location_names[0]
            obs, reward, done = execute_push_switch(env, task, switch, location)
            print("[Exploration] Light switch action executed.")
        except Exception as e:
            print("[Exploration] Failed to execute_push_switch:", e)

        # Try to pick up each object (execute_pick)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick up {obj} (execute_pick)...")
                obs, reward, done = execute_pick(env, task, obj, location_names[0])
                print(f"[Exploration] Picked up {obj}.")
                # Try to place the object in a drawer if possible
                try:
                    print(f"[Exploration] Attempting to place {obj} in {drawer_names[0]} (execute_place)...")
                    obs, reward, done = execute_place(env, task, obj, drawer_names[0], location_names[0])
                    print(f"[Exploration] Placed {obj} in {drawer_names[0]}.")
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer_names[0]}:", e)
            except Exception as e:
                print(f"[Exploration] Failed to pick up {obj}:", e)

        # Try to open and close drawers (execute_pull, execute_push)
        for drawer in drawer_names:
            for handle in handle_names:
                try:
                    print(f"[Exploration] Attempting to pull {drawer} with {handle} (execute_pull)...")
                    obs, reward, done = execute_pull(env, task, drawer, handle, location_names[0])
                    print(f"[Exploration] Pulled {drawer} with {handle}.")
                except Exception as e:
                    print(f"[Exploration] Failed to pull {drawer} with {handle}:", e)
            try:
                print(f"[Exploration] Attempting to push (close) {drawer} (execute_push)...")
                obs, reward, done = execute_push(env, task, drawer, location_names[0])
                print(f"[Exploration] Pushed (closed) {drawer}.")
            except Exception as e:
                print(f"[Exploration] Failed to push (close) {drawer}:", e)

        # Try to move between locations (execute_go)
        if len(location_names) > 1:
            try:
                print(f"[Exploration] Attempting to move from {location_names[0]} to {location_names[1]} (execute_go)...")
                obs, reward, done = execute_go(env, task, location_names[0], location_names[1])
                print(f"[Exploration] Moved from {location_names[0]} to {location_names[1]}.")
            except Exception as e:
                print(f"[Exploration] Failed to move between locations:", e)

        # Try to sweep objects (execute_sweep)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to sweep {obj} (execute_sweep)...")
                obs, reward, done = execute_sweep(env, task, obj, location_names[0])
                print(f"[Exploration] Swept {obj}.")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}:", e)

        # Try to use gripper (execute_gripper)
        try:
            print("[Exploration] Attempting to use gripper (execute_gripper)...")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] Gripper action executed.")
        except Exception as e:
            print("[Exploration] Failed to execute_gripper:", e)

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()