# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and confirm the robot's location predicates.

        # Try to use execute_go to move the robot and observe the effect
        # Since we don't have the exact object/room names, we will attempt to use the available positions

        # Find all locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'location' in name:
                location_names.append(name)
        # If no explicit location names, just use all keys
        if not location_names:
            location_names = list(positions.keys())

        # For exploration, try to move the robot between two locations if possible
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            print(f"[Exploration] Attempting to move robot from {from_loc} to {to_loc} using execute_go.")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[from_loc],
                    to_location=positions[to_loc],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_go: {e}")
        else:
            print("[Exploration] Not enough locations found to perform execute_go.")

        # After moving, try to pick an object if possible to further explore predicates
        # Find an object on the floor
        object_on_floor = None
        for name in positions:
            if 'object' in name or 'ball' in name or 'handle' in name:
                object_on_floor = name
                break
        if object_on_floor and len(location_names) > 0:
            print(f"[Exploration] Attempting to pick {object_on_floor} at {location_names[0]} using execute_pick.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_on_floor,
                    location_name=location_names[0],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {object_on_floor} at {location_names[0]}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_pick: {e}")
        else:
            print("[Exploration] No suitable object found to pick.")

        # Try to place the object in a drawer if possible
        drawer_name = None
        for name in positions:
            if 'drawer' in name:
                drawer_name = name
                break
        if object_on_floor and drawer_name and len(location_names) > 0:
            print(f"[Exploration] Attempting to place {object_on_floor} in {drawer_name} at {location_names[0]} using execute_place.")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=object_on_floor,
                    drawer_name=drawer_name,
                    location_name=location_names[0],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Placed {object_on_floor} in {drawer_name} at {location_names[0]}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_place: {e}")
        else:
            print("[Exploration] No suitable drawer found to place object.")

        # Try to push or pull the drawer to further explore predicates
        if drawer_name and len(location_names) > 0:
            print(f"[Exploration] Attempting to push {drawer_name} at {location_names[0]} using execute_push.")
            try:
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location_name=location_names[0],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Pushed {drawer_name} at {location_names[0]}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_push: {e}")

            # Try to pull the drawer (requires a handle object)
            handle_name = None
            for name in positions:
                if 'handle' in name:
                    handle_name = name
                    break
            if handle_name:
                print(f"[Exploration] Attempting to pull {drawer_name} with {handle_name} at {location_names[0]} using execute_pull.")
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer_name,
                        handle_name=handle_name,
                        location_name=location_names[0],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Pulled {drawer_name} with {handle_name} at {location_names[0]}.")
                except Exception as e:
                    print(f"[Exploration] Failed to execute_pull: {e}")
            else:
                print("[Exploration] No suitable handle found to pull drawer.")

        # Try to push the switch to turn on the light if available
        switch_name = None
        for name in positions:
            if 'switch' in name or 'light' in name:
                switch_name = name
                break
        if switch_name and len(location_names) > 0:
            print(f"[Exploration] Attempting to push switch {switch_name} at {location_names[0]} using execute_push_switch.")
            try:
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=switch_name,
                    location_name=location_names[0],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Pushed switch {switch_name} at {location_names[0]}.")
            except Exception as e:
                print(f"[Exploration] Failed to execute_push_switch: {e}")
        else:
            print("[Exploration] No suitable switch found to push.")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()