# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will attempt to use available skills to explore and confirm the robot's location.

        # Try to find the robot's current location among known locations
        robot_location = None
        possible_locations = []
        for name, pos in positions.items():
            if 'floor' in name.lower():
                possible_locations.append(name)
            if 'robot' in name.lower():
                robot_location = name

        # If we can't find a robot object, assume the robot starts at 'floor' or the first location
        if not robot_location:
            robot_location = 'robot'
        if not possible_locations:
            # Fallback: use all location names from positions
            possible_locations = [name for name in positions.keys() if 'location' in name.lower() or 'floor' in name.lower()]

        # For exploration, try to move the robot to each possible location using execute_go
        # and check if the predicate (robot-at <location>) is satisfied.
        # Since we cannot check predicates directly, we log the actions.

        print("[Exploration] Attempting to move robot to each possible location to identify (robot-at <loc>) predicate.")
        for loc in possible_locations:
            try:
                print(f"[Exploration] Trying to move robot to location: {loc}")
                # Use execute_go if available
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {loc}.")
                robot_location = loc  # Update robot's current location
                if done:
                    print("[Exploration] Task ended during exploration phase.")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # After exploration, we assume the missing predicate is (robot-at floor)
        print("[Exploration] Exploration phase complete. Missing predicate likely (robot-at floor).")

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan using only the available skills.
        # For demonstration, we show a generic sequence using the available skills.

        # Example: Turn on the light if the room is dark
        try:
            print("[Task] Attempting to turn on the light (execute_push_switch).")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name='switch',
                location=robot_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Light turned on.")
            if done:
                print("[Task] Task ended after turning on the light.")
                return
        except Exception as e:
            print(f"[Task] Failed to turn on the light: {e}")

        # Example: Pick up an object on the floor
        for obj_name in positions.keys():
            if 'object' in obj_name.lower() or 'ball' in obj_name.lower():
                try:
                    print(f"[Task] Attempting to pick up {obj_name} at {robot_location}.")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj_name,
                        location=robot_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Task] Picked up {obj_name}.")
                    if done:
                        print("[Task] Task ended after picking up object.")
                        return
                except Exception as e:
                    print(f"[Task] Failed to pick up {obj_name}: {e}")

        # Example: Place the object in a drawer if available
        for drawer_name in positions.keys():
            if 'drawer' in drawer_name.lower():
                try:
                    print(f"[Task] Attempting to place object in {drawer_name} at {robot_location}.")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj_name,
                        drawer_name=drawer_name,
                        location=robot_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Task] Placed object in {drawer_name}.")
                    if done:
                        print("[Task] Task ended after placing object.")
                        return
                except Exception as e:
                    print(f"[Task] Failed to place object in {drawer_name}: {e}")

        # Additional steps (pull, push, sweep, gripper) can be added as needed using the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()