# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the primitives defined here

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        ##################################################
        # === EXPLORATION PHASE FOR MISSING PREDICATES ===
        ##################################################
        # Guided by feedback and the exploration domain definition,
        # we try available skills to gather knowledge about the state,
        # e.g., which predicate is missing (identified, temperature-known, lock-known, etc.)
        #
        # We proceed to explore the effects and observation using each possible skill:
        
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Try to gather all objects, drawers, and locations from positions (assumed structure)
        # Example positions: {
        #     'object_1': {'type':'object', 'pos':(...), 'location':'room1'},
        #     'drawer_1': {'type':'drawer', ...}
        # }
        object_names = [name for name, props in positions.items() if props.get('type') == 'object']
        drawer_names = [name for name, props in positions.items() if props.get('type') == 'drawer']
        location_names = list({props.get('location') for props in positions.values() if 'location' in props})
        
        # For the sake of generality, fallback to some default sample names if extraction fails
        if not object_names:
            object_names = ['object_1']
        if not drawer_names:
            drawer_names = ['drawer_1']
        if not location_names:
            location_names = ['loc_1', 'loc_2']

        # Track progress
        exploration_results = []

        print("\n[Exploration] Starting exploration phase to determine missing predicates.\n")

        # Try moving to all locations and track observations
        try:
            for i, from_loc in enumerate(location_names):
                for j, to_loc in enumerate(location_names):
                    if from_loc != to_loc:
                        try:
                            print(f"[Exploration] execute_go: Moving from {from_loc} to {to_loc}")
                            obs, reward, done = execute_go(env, task, from_loc, to_loc)
                            exploration_results.append(
                                {'action': 'execute_go', 'from': from_loc, 'to': to_loc, 'obs': obs}
                            )
                            if done:
                                print("[Exploration] Task ended during location exploration.")
                                return
                        except Exception as e:
                            print(f"[WARNING] execute_go failed from {from_loc} to {to_loc}: {e}")

        except Exception as e:
            print(f"[ERROR] Exploration phase failed during 'execute_go': {e}")

        # Try picking all objects at their known locations (simulate picking and observe state)
        try:
            for obj in object_names:
                obj_props = positions.get(obj, {})
                loc = obj_props.get('location', location_names[0])
                try:
                    print(f"[Exploration] execute_pick: Picking {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    exploration_results.append(
                        {'action': 'execute_pick', 'object': obj, 'location': loc, 'obs': obs}
                    )
                    if done:
                        print("[Exploration] Task ended during pick exploration.")
                        return
                except Exception as e:
                    print(f"[WARNING] execute_pick failed for {obj} at {loc}: {e}")
        except Exception as e:
            print(f"[ERROR] Exploration phase failed during 'execute_pick': {e}")

        # Try pulling all drawers with all possible handles
        # Handles are objects associated to drawers via 'handle-of' predicate, we guess names
        for drawer in drawer_names:
            for handle_candidate in object_names:
                loc = positions.get(drawer, {}).get('location', location_names[0])
                try:
                    print(f"[Exploration] execute_pull: Trying to pull {drawer} with handle {handle_candidate} at {loc}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_candidate, loc)
                    exploration_results.append(
                        {
                            'action': 'execute_pull',
                            'drawer': drawer,
                            'handle': handle_candidate,
                            'location': loc,
                            'obs': obs
                        }
                    )
                    if done:
                        print("[Exploration] Task ended during pull exploration.")
                        return
                except Exception as e:
                    print(f"[WARNING] execute_pull failed for {drawer} and handle {handle_candidate}: {e}")

        # Try opening each drawer (execute_push) and placing objects (execute_place)
        for drawer in drawer_names:
            loc = positions.get(drawer, {}).get('location', location_names[0])
            try:
                print(f"[Exploration] execute_push: Trying to push (close) drawer {drawer} at {loc}")
                obs, reward, done = execute_push(env, task, drawer, loc)
                exploration_results.append(
                    {'action': 'execute_push', 'drawer': drawer, 'location': loc, 'obs': obs}
                )
                if done:
                    print("[Exploration] Task ended during push exploration.")
                    return
            except Exception as e:
                print(f"[WARNING] execute_push failed for {drawer} at {loc}: {e}")

            for obj in object_names:
                try:
                    print(f"[Exploration] execute_place: Trying to place {obj} into drawer {drawer} at {loc}")
                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    exploration_results.append(
                        {'action': 'execute_place', 'object': obj, 'drawer': drawer, 'location': loc, 'obs': obs}
                    )
                    if done:
                        print("[Exploration] Task ended during place exploration.")
                        return
                except Exception as e:
                    print(f"[WARNING] execute_place failed for {obj} into {drawer} at {loc}: {e}")

        # Try the rest of the available actions, generically
        for skill in available_skills:
            if skill in ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_go']:
                continue  # Already tried above
            try:
                # Try each action with plausible arguments as possible
                if skill == 'execute_sweep':
                    for obj in object_names:
                        loc = positions.get(obj, {}).get('location', location_names[0])
                        print(f"[Exploration] execute_sweep: Sweeping {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        exploration_results.append(
                            {'action': 'execute_sweep', 'object': obj, 'location': loc, 'obs': obs}
                        )
                        if done:
                            print("[Exploration] Task ended during sweep exploration.")
                            return
                elif skill == 'execute_rotate':
                    # If defined, would add code here (no parameters documented in the main PDDL)
                    pass
                elif skill == 'execute_gripper':
                    print("[Exploration] execute_gripper")
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results.append(
                        {'action': 'execute_gripper', 'obs': obs}
                    )
                    if done:
                        print("[Exploration] Task ended during gripper exploration.")
                        return
            except Exception as e:
                print(f"[WARNING] {skill} failed: {e}")

        print("\n[Exploration] Exploration phase complete. Review environment observation and history to infer missing predicates or state.\n")
        # Optionally, print all exploration results
        # for res in exploration_results:
        #     print(res)

        ##############################
        # === (Optional) MAIN TASK ===
        ##############################
        # Once the missing predicate is discovered, the main plan can be executed.
        # The main task plan execution code would go here.
        # For this assignment, only the exploration phase is required to be completed.
        # You can add your full plan execution logic below if needed, using the same
        # skill function calls as in exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
