# ====== Generated Executable Code (after [Frozen Code End]) ======

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock a cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'bin'
        # ]

        # For this task, we will:
        # 1. Move gripper to the side of the bottom drawer (bottom_side_pos)
        # 2. Rotate gripper to 90 degrees (for grasping drawer handle)
        # 3. Move gripper to the anchor position of the bottom drawer (bottom_anchor_pos)
        # 4. Pick (grasp) the drawer handle at anchor position
        # 5. Pull the drawer open (pull along x axis)
        # 6. Move gripper above the trash (item3) on the table
        # 7. Pick the trash (item3)
        # 8. Move gripper above the bin
        # 9. Place the trash in the bin

        # --- Step 1: Move to side of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_side_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle) ---
        # We'll assume 90 deg is [0, 0, sin(45°), cos(45°)] in quaternion (z axis)
        # RLBench uses [x, y, z, w] order
        angle_rad = np.deg2rad(90)
        target_quat = np.array([0, 0, np.sin(angle_rad/2), np.cos(angle_rad/2)])
        print("[Task] Rotating gripper to 90 degrees (z axis)")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        print("[Task] Grasping drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open (pull along x axis) ---
        # We'll pull by 0.15 meters along x axis (positive x)
        print("[Task] Pulling the drawer open along x axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # --- Step 6: Move above the trash (item3) on the table ---
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 not found in object positions.")
            shutdown_environment(env)
            return

        # Raise z a bit for approach
        item3_above = np.array(item3_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above item3 (trash):", item3_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=item3_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # --- Step 7: Pick the trash (item3) ---
        print("[Task] Picking up item3 (trash)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(item3_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item3)!")
            return

        # --- Step 8: Move above the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in object positions.")
            shutdown_environment(env)
            return

        bin_above = np.array(bin_pos) + np.array([0, 0, 0.15])
        print("[Task] Moving above bin:", bin_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # --- Step 9: Place the trash in the bin ---
        print("[Task] Placing item3 (trash) in the bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(bin_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
