import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan Execution ===
        # Goal: Pull open any drawer that is not locked, then drop the rubbish into the bin.

        # 1. Select a drawer to open (bottom, middle, or top)
        # We'll choose 'bottom' drawer for this plan.
        drawer_name = 'bottom'
        gripper_name = 'gripper'
        # Positions for the drawer
        side_pos_name = 'bottom_side_pos'
        anchor_pos_name = 'bottom_anchor_pos'
        # The joint position is not used directly, but could be used for more advanced logic.

        # 2. Get required positions
        try:
            # Drawer handle side and anchor positions
            side_pos = positions[side_pos_name]
            anchor_pos = positions[anchor_pos_name]
            # For pulling, we need to know the pull direction and distance.
            # We'll assume pulling along +x by 0.15m (typical drawer pull distance).
            pull_axis = 'x'
            pull_distance = 0.15
        except Exception as e:
            print(f"[Error] Could not retrieve drawer positions: {e}")
            shutdown_environment(env)
            return

        # 3. Get rubbish and bin positions
        try:
            rubbish_name = 'item3'  # According to object list, 'item3' is the rubbish
            rubbish_pos = positions[rubbish_name]
            bin_pos = positions['bin']
        except Exception as e:
            print(f"[Error] Could not retrieve rubbish or bin positions: {e}")
            shutdown_environment(env)
            return

        # 4. Step 1: Move gripper to the side position of the drawer (approach handle)
        print(f"[Plan] Step 1: Move to drawer side position ({side_pos_name}) at {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended after move to drawer side position!")
            return

        # 5. Step 2: Rotate gripper to 90 degrees (vertical for grasping handle)
        print("[Plan] Step 2: Rotate gripper to 90 degrees (vertical)")
        # We'll assume 'ninety_deg' means rotation about z by 90 degrees.
        # Get current orientation and compute target quaternion for 90 deg about z.
        from scipy.spatial.transform import Rotation as R
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        # Compose with 90 deg about z
        rot_90z = R.from_euler('z', 90, degrees=True)
        current_rot = R.from_quat(current_quat)
        target_rot = rot_90z * current_rot
        target_quat = target_rot.as_quat()
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # 6. Step 3: Move to anchor position (in front of drawer handle)
        print(f"[Plan] Step 3: Move to drawer anchor position ({anchor_pos_name}) at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # 7. Step 4: Pull the drawer open
        print(f"[Plan] Step 4: Pull the drawer along {pull_axis} by {pull_distance}m")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Task] Task ended after pull!")
            return

        # 8. Step 5: Move to rubbish position (item3)
        print(f"[Plan] Step 5: Move to rubbish position ({rubbish_name}) at {rubbish_pos}")
        obs, reward, done = move(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Task ended after move to rubbish!")
            return

        # 9. Step 6: Pick the rubbish
        print(f"[Plan] Step 6: Pick the rubbish ({rubbish_name})")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after pick!")
            return

        # 10. Step 7: Move to bin position
        print(f"[Plan] Step 7: Move to bin position at {bin_pos}")
        obs, reward, done = move(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # 11. Step 8: Place the rubbish in the bin
        print(f"[Plan] Step 8: Place the rubbish in the bin")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()