import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Select a drawer and open it fully, then pick up the rubbish and leave it in the trash can.

        # --- Step 1: Rotate gripper to 90 degrees (vertical for drawer manipulation) ---
        # We'll use the 'rotate' skill to set the gripper orientation.
        # Assume 'ninety_deg' means gripper z-axis is vertical (up), which is [0, 0, 0.7071, 0.7071] in quaternion (90deg about x).
        # We'll use the current position and set the orientation.
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_pos = obs.gripper_pose[:3]
            # 90 degrees about x-axis (RLBench uses xyzw)
            target_quat = np.array([0.7071, 0, 0, 0.7071])
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Rotate failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to the side position of the bottom drawer ---
        # We'll use 'bottom_side_pos' as the approach point for the bottom drawer.
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Move to side position failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to the anchor position of the bottom drawer ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Move to anchor position failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Move gripper to the joint position of the bottom drawer (simulate grasp) ---
        # This is a small adjustment to ensure the gripper is at the drawer handle.
        try:
            joint_pos = positions.get('bottom_joint_pos', None)
            if joint_pos is None:
                raise KeyError("bottom_joint_pos not found in positions.")
            obs, reward, done = move(env, task, joint_pos)
            if done:
                print("[Task] Task ended after move to joint position!")
                return
        except Exception as e:
            print(f"[ERROR] Move to joint position failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        # We'll pull along the x-axis (assume drawer opens along +x, adjust if needed).
        try:
            # Pull distance: estimate 0.18m (adjust as needed for full open)
            pull_distance = 0.18
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Pull failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up the rubbish from the table (now accessible) ---
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise KeyError("rubbish not found in positions.")
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[ERROR] Pick failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Place failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()