import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Unlock the cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.

        # Object name mapping (from object list and positions)
        # We'll use 'middle' drawer as the accessible one (not locked, or at least not 'top' which is locked)
        # Let's assume 'middle_anchor_pos' is the anchor position for the middle drawer
        # 'item3' is the trash (rubbish), 'bin' is the trash bin

        # Defensive: check for required keys
        required_keys = [
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'item3', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Step 1: Move gripper to the side position of the middle drawer (preparing to rotate)
        print("[Plan] Step 1: Move to middle_side_pos (side of middle drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['middle_side_pos'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to middle_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (required for grasping drawer handle)
        print("[Plan] Step 2: Rotate gripper to 90 degrees (xyzw for 90deg about z)")
        # We'll assume the gripper's z-axis rotation is required; get current pos, set target quat
        # 90 degrees about z: quaternion [x, y, z, w] = [0, 0, sin(θ/2), cos(θ/2)] for θ=90deg
        theta_rad = np.deg2rad(90)
        target_quat = np.array([0, 0, np.sin(theta_rad/2), np.cos(theta_rad/2)])
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position of the middle drawer (handle position)
        print("[Plan] Step 3: Move to middle_anchor_pos (anchor/handle of middle drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['middle_anchor_pos'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to middle_anchor_pos!")
            return

        # Step 4: Pick (grasp) the drawer handle (simulate as pick at anchor position)
        print("[Plan] Step 4: Pick (grasp) the middle drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=positions['middle_anchor_pos'],
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # Step 5: Pull the drawer open (pull along x axis, e.g., 0.15m)
        print("[Plan] Step 5: Pull the middle drawer open along x axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (open drawer)!")
            return

        # Step 6: Move to the trash (item3) on the table
        print("[Plan] Step 6: Move to item3 (trash) on the table")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['item3'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # Step 7: Pick the trash (item3)
        print("[Plan] Step 7: Pick the trash (item3)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=positions['item3'],
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item3)!")
            return

        # Step 8: Move to the bin
        print("[Plan] Step 8: Move to bin")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['bin'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 9: Place the trash in the bin
        print("[Plan] Step 9: Place the trash (item3) in the bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=positions['bin'],
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[ERROR] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()