import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then drop the rubbish into the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Step 1: Pull open any unlocked drawer ===
        # From the initial state, 'bottom' drawer is not locked, 'middle' and 'top' are locked.
        # We'll use the 'bottom' drawer.

        # Get relevant positions for the bottom drawer
        # Convention: anchor_pos is where to grasp, side_pos is where to approach, joint_pos is the pull direction reference
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
            bottom_side_pos = positions['bottom_side_pos']
            bottom_joint_pos = positions['bottom_joint_pos']
        except KeyError as e:
            print(f"[Error] Missing position for bottom drawer: {e}")
            shutdown_environment(env)
            return

        # Approach the side position first (optional, for safety)
        try:
            print("[Task] Moving to bottom drawer side position...")
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended unexpectedly during move to side position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # Move to anchor position to grasp the drawer handle
        try:
            print("[Task] Moving to bottom drawer anchor position...")
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended unexpectedly during move to anchor position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # Grasp the drawer handle (simulate pick on the anchor position)
        try:
            print("[Task] Grasping the bottom drawer handle...")
            obs, reward, done = pick(
                env,
                task,
                target_pos=bottom_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended unexpectedly during pick (grasp drawer handle)!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick (drawer handle): {e}")
            shutdown_environment(env)
            return

        # Pull the drawer open: pull along the x-axis (assume positive x is the pull direction)
        # Estimate pull distance (e.g., 0.15m)
        pull_distance = 0.15
        try:
            print("[Task] Pulling the bottom drawer open...")
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended unexpectedly during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # === Step 2: Pick up the rubbish ===
        # Find the rubbish object (item3 is the rubbish per object list and initial state)
        try:
            rubbish_pos = positions['item3']
        except KeyError as e:
            print(f"[Error] Missing position for rubbish (item3): {e}")
            shutdown_environment(env)
            return

        try:
            print("[Task] Picking up the rubbish (item3)...")
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended unexpectedly during pick (rubbish)!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick (rubbish): {e}")
            shutdown_environment(env)
            return

        # === Step 3: Place the rubbish into the bin ===
        try:
            bin_pos = positions['bin']
        except KeyError as e:
            print(f"[Error] Missing position for bin: {e}")
            shutdown_environment(env)
            return

        try:
            print("[Task] Placing the rubbish into the bin...")
            obs, reward, done = place(
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Exception during place (bin): {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()