import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Pull open any drawer that is not locked, then drop the rubbish into the bin.

        # 1. Find an unlocked drawer (bottom drawer is not locked in the initial state)
        unlocked_drawer = None
        drawer_names = ['bottom', 'middle', 'top']
        for drawer in drawer_names:
            # In the initial state, only 'bottom' is not locked
            if drawer == 'bottom':
                unlocked_drawer = drawer
                break
        if unlocked_drawer is None:
            print("[Task] No unlocked drawer found! Aborting.")
            return

        # 2. Get relevant positions for the drawer
        # Convention: positions dict keys are e.g. 'bottom_anchor_pos', 'bottom_side_pos'
        anchor_pos_key = f"{unlocked_drawer}_anchor_pos"
        side_pos_key = f"{unlocked_drawer}_side_pos"
        joint_pos_key = f"{unlocked_drawer}_joint_pos"
        try:
            anchor_pos = positions[anchor_pos_key]
            side_pos = positions[side_pos_key]
            joint_pos = positions[joint_pos_key]
        except KeyError as e:
            print(f"[Task] Missing position for {e}. Aborting.")
            return

        # 3. Step 1: Rotate gripper to 90 degrees (vertical for grasping drawer handle)
        # We'll use a quaternion for 90 deg rotation about z axis
        # RLBench uses [x, y, z, w] convention
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # 4. Step 2: Move to the side position of the drawer (approach handle from the side)
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Task] Task ended during move to side!")
            return

        # 5. Step 3: Move to the anchor position (align with drawer handle)
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Task ended during move to anchor!")
            return

        # 6. Step 4: Move to the joint position (simulate grasping the handle)
        obs, reward, done = move(env, task, target_pos=np.array(joint_pos))
        if done:
            print("[Task] Task ended during move to joint!")
            return

        # 7. Step 5: Pull the drawer open (pull along x axis, e.g. 0.15m)
        # We'll assume the drawer opens along the positive x axis
        pull_distance = 0.15
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull!")
            return

        # 8. Step 6: Pick up the rubbish from the table
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Task] Rubbish position not found! Aborting.")
            return
        obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos), approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during pick!")
            return

        # 9. Step 7: Place the rubbish into the bin
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Task] Bin position not found! Aborting.")
            return
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos), approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()