import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: bottom_anchor_pos, bottom_joint_pos, bottom_side_pos, middle_anchor_pos, middle_joint_pos, middle_side_pos, top_anchor_pos, top_joint_pos, top_side_pos, item1, item2, item3, bin

        # --- Plan: Open a drawer fully, pick up the rubbish, and place it in the bin ---

        # For this plan, we will:
        # 1. Move gripper to the side position of the bottom drawer (for handle approach)
        # 2. Rotate gripper to 90 degrees (to align with handle)
        # 3. Move to anchor position (for grasping the handle)
        # 4. Pick the drawer handle (simulate grasp)
        # 5. Pull the drawer open (pull along x axis)
        # 6. Move gripper above the rubbish (item3)
        # 7. Rotate gripper to 0 degrees (for vertical grasp)
        # 8. Move down to the rubbish
        # 9. Pick the rubbish
        # 10. Move above the bin
        # 11. Place the rubbish in the bin

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_side_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (align with handle) ---
        # Assume 90 deg about z axis from initial orientation
        quat_90deg = [0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]  # xyzw, 90deg about z
        print("[Task] Rotating gripper to 90 degrees (about z axis)")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=np.array(quat_90deg),
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate to 90deg!")
            return

        # --- Step 3: Move to anchor position of bottom drawer (for grasp) ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer anchor position:", bottom_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        # Use pick with approach from above (z axis)
        print("[Task] Grasping the drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open (pull along x axis, e.g. 0.18m) ---
        print("[Task] Pulling the drawer open along x axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.18,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # --- Step 6: Move above the rubbish (item3) ---
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 not found in object positions.")
            shutdown_environment(env)
            return

        # Move above item3 (add offset in z for approach)
        item3_above = np.array(item3_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above the rubbish (item3):", item3_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=item3_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # --- Step 7: Rotate gripper to 0 degrees (vertical for pick) ---
        quat_0deg = [0, 0, 0, 1]  # identity quaternion
        print("[Task] Rotating gripper to 0 degrees (vertical)")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=np.array(quat_0deg),
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate to 0deg!")
            return

        # --- Step 8: Move down to the rubbish (item3) ---
        print("[Task] Moving down to the rubbish (item3):", item3_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(item3_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # --- Step 9: Pick the rubbish (item3) ---
        print("[Task] Picking up the rubbish (item3)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(item3_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item3)!")
            return

        # --- Step 10: Move above the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in object positions.")
            shutdown_environment(env)
            return

        bin_above = np.array(bin_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above the bin:", bin_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # --- Step 11: Place the rubbish in the bin ---
        print("[Task] Placing the rubbish (item3) in the bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(bin_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()