import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===

        # Step 1: Move to the side position of the bottom drawer
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_side_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        from scipy.spatial.transform import Rotation as R
        quat_90_z = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Task] Rotating gripper to 90 deg about z axis (drawer grasp orientation).")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90_z,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position of bottom drawer
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        print("[Task] Grasping the bottom drawer handle at anchor position.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='y',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # Step 5: Pull the drawer open
        pull_distance = 0.15
        print(f"[Task] Pulling the bottom drawer open by {pull_distance}m along +x axis.")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # Step 6: Move to the trash (item3)
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 (rubbish) not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to trash (item3) at:", item3_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(item3_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to trash!")
            return

        # Step 7: Pick up the trash (item3)
        print("[Task] Picking up trash (item3).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(item3_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (trash)!")
            return

        # Step 8: Move to the bin
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bin at:", bin_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bin_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 9: Place the trash in the bin
        print("[Task] Placing trash (item3) in bin at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(bin_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()