# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'bin'
        # ]
        # We'll use the bottom drawer for this plan.

        # Plan:
        # 1. Move to the side position of the bottom drawer.
        # 2. Rotate gripper to 90 degrees (for side grasp).
        # 3. Move to anchor position of the bottom drawer.
        # 4. Pick (grasp) the drawer handle at anchor position.
        # 5. Pull the drawer open.
        # 6. Move away from the drawer (to nowhere or safe pos).
        # 7. Rotate gripper back to 0 degrees.
        # 8. Move to above item3 (rubbish) on the table.
        # 9. Pick item3.
        # 10. Move to bin.
        # 11. Place item3 in bin.

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in positions.")
            return
        print("[Step 1] Move to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (side grasp) ---
        # Assume 90 deg is [x, y, z, w] quaternion for 90 deg about z axis
        # We'll use scipy to get the quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Step 2] Rotate gripper to 90 degrees (z axis):", quat_90deg)
        obs, reward, done = rotate(env, task, target_quat=quat_90deg)
        if done:
            print("[Task] Task ended after rotate to 90 deg!")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in positions.")
            return
        print("[Step 3] Move to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        print("[Step 4] Pick (grasp) drawer at anchor position.")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick (grasp) drawer!")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along x axis by a fixed distance (e.g., 0.15m)
        print("[Step 5] Pull the drawer open (along x axis).")
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 6: Move away from drawer (to nowhere_pos for safety) ---
        try:
            nowhere_pos = positions['bottom_joint_pos']  # Use joint pos as a safe "away" pos
        except KeyError:
            nowhere_pos = bottom_side_pos  # fallback
        print("[Step 6] Move away from drawer to:", nowhere_pos)
        obs, reward, done = move(env, task, target_pos=nowhere_pos)
        if done:
            print("[Task] Task ended after move away from drawer!")
            return

        # --- Step 7: Rotate gripper back to 0 degrees ---
        quat_0deg = R.from_euler('z', 0, degrees=True).as_quat()
        print("[Step 7] Rotate gripper back to 0 degrees:", quat_0deg)
        obs, reward, done = rotate(env, task, target_quat=quat_0deg)
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # --- Step 8: Move to above item3 (rubbish) on the table ---
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 not found in positions.")
            return
        # Approach from above (z axis)
        item3_above = np.array(item3_pos) + np.array([0, 0, 0.10])
        print("[Step 8] Move to above item3:", item3_above)
        obs, reward, done = move(env, task, target_pos=item3_above)
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # --- Step 9: Pick item3 (rubbish) ---
        print("[Step 9] Pick item3 (rubbish).")
        obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # --- Step 10: Move to bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in positions.")
            return
        # Approach from above
        bin_above = np.array(bin_pos) + np.array([0, 0, 0.10])
        print("[Step 10] Move to above bin:", bin_above)
        obs, reward, done = move(env, task, target_pos=bin_above)
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # --- Step 11: Place item3 in bin ---
        print("[Step 11] Place item3 in bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print("[Exception] An error occurred during task execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
