import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: bottom_anchor_pos, bottom_joint_pos, bottom_side_pos, middle_anchor_pos, middle_joint_pos, middle_side_pos, top_anchor_pos, top_joint_pos, top_side_pos, item1, item2, item3, bin

        # --- PLAN: Open a drawer fully, pick up the rubbish, and place it in the bin ---

        # Step 1: Move to the side position of the bottom drawer
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return

        # Step 2: Prepare quaternion for 90 degrees rotation about z axis
        ninety_deg_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z

        # Step 3: Move to anchor position of the bottom drawer
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return

        # Step 4: Set pull distance for drawer opening
        pull_distance = 0.20

        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        obs, reward, done = move(env, task, target_pos=approach_above_item3)
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after opening the drawer!")
            return

        # Step 6: Move to the table (where rubbish is)
        # Let's use item3 as the rubbish (per object list)
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 not found in object positions.")
            shutdown_environment(env)
            return

        # Move above the item3 (rubbish)
        approach_above_item3 = np.array(item3_pos) + np.array([0, 0, 0.10])
        print("[Step 6] Move above the rubbish (item3):", approach_above_item3)
        obs, reward, done = move(env, task, target_pos=approach_above_item3)
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # Step 7: Rotate gripper to 0 degrees (top grasp)
        zero_deg_quat = np.array([0, 0, 0, 1])  # Identity quaternion
        print("[Step 7] Rotate gripper to 0 degrees (top grasp)")
        obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # Step 8: Move directly above the rubbish (item3)
        print("[Step 8] Move to precise pick position of item3:", item3_pos)
        obs, reward, done = move(env, task, target_pos=item3_pos)
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # Step 9: Pick the rubbish (item3)
        print("[Step 9] Pick the rubbish (item3)")
        obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # Step 10: Move above the bin
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in object positions.")
            shutdown_environment(env)
            return

        approach_above_bin = np.array(bin_pos) + np.array([0, 0, 0.10])
        print("[Step 10] Move above the bin:", approach_above_bin)
        obs, reward, done = move(env, task, target_pos=approach_above_bin)
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # Step 11: Place the rubbish in the bin
        print("[Step 11] Place the rubbish (item3) in the bin")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print("[Exception] An error occurred during task execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()