import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: Move gripper to the side position of the bottom drawer
        try:
            bottom_side_pos = positions['bottom_side_pos']
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError as e:
            print(f"[Error] '{e.args[0]}' not found in object positions.")
            shutdown_environment(env)
            return

        # Prepare quaternion for 90 degree rotation about z axis
        # RLBench/pyrep uses [x, y, z, w] (xyzw) format
        # 90 deg about z: [0, 0, sin(45°), cos(45°)] = [0, 0, 0.7071, 0.7071]
        quat_90deg = np.array([0.0, 0.0, 0.70710678, 0.70710678])

        # [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = rotate(
                env,
                task,
                target_quat=quat_90deg,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = pick(
                env,
                task,
                target_pos=bottom_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = pull(
                env,
                task,
                pull_distance=0.15,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = place(
            env,
            task,
            target_pos=bottom_side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after opening drawer!")
            return

        # Step 8: Move gripper to item1 on the table (rubbish)
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] 'item1' not found in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(
            env,
            task,
            target_pos=item1_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # Step 9: Pick item1 (rubbish) from table
        obs, reward, done = pick(
            env,
            task,
            target_pos=item1_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 10: Move gripper to bin
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] 'bin' not found in object positions.")
            shutdown_environment(env)
            return

        obs, reward, done = move(
            env,
            task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 11: Place item1 in bin
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed after placing item1 in bin! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()