# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put the rubbish in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'bin'

        # --- Plan: Open a drawer (bottom), then put rubbish in bin ---

        # 1. Rotate gripper to 90 degrees (vertical) for side approach
        #    - Use bottom drawer for this plan
        try:
            ninety_deg_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # z-axis 90deg
            [Frozen Code Start]
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            [Frozen Code End]
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # 2. Move to side position of bottom drawer
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=side_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # 3. Move to anchor position of bottom drawer (handle)
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # 4. Pick (grasp) the drawer handle at anchor position
        try:
            # For this domain, the "pick" skill is used for objects, but for the drawer handle, we simulate grasp by moving to anchor and closing gripper (already at anchor).
            # So, we use pick with the anchor position to close the gripper on the handle.
            obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.04, approach_axis='y')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # 5. Pull the drawer open (along x axis, e.g., 0.12m)
        try:
            pull_distance = 0.12  # meters, adjust as needed for your environment
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # 6. Pick up the rubbish from the table
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise KeyError("rubbish not found in positions.")
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # 7. Place the rubbish in the bin
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()