import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer to unlock the cabinet, then throw away the trash, leaving other objects untouched.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object/Position Key Mapping ---
        # Drawer positions (assume keys in positions dict)
        gripper_home_pos = positions.get('waypoint1', None)
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        # For rubbish
        rubbish_pos = positions.get('rubbish', None)
        # For bin
        bin_pos = positions.get('bin', None)

        # === PLAN EXECUTION ===
        # Step 1: Rotate gripper to 90 degrees (required for side approach)
        # Step 2: Move to side position of bottom drawer
        # Step 3: Move to anchor position of bottom drawer
        # Step 4: Pick (grasp) the drawer handle at anchor position
        # Step 5: Pull the drawer open
        # Step 6: Pick up the rubbish from the table
        # Step 7: Place the rubbish in the bin

        # --- Step 1: Rotate gripper to 90 degrees (z axis) ---
        # Assume 'ninety_deg' means gripper points sideways (for drawer)
        # We'll use a quaternion for 90 deg rotation about z axis
        # RLBench uses [x, y, z, w] (xyzw) format
        # 90 deg about z: [0, 0, sin(45°), cos(45°)] = [0, 0, 0.7071, 0.7071]
        target_quat_ninety = np.array([0, 0, 0.7071, 0.7071])
        try:
            obs, reward, done = rotate(env, task, target_quat=target_quat_ninety)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Rotate failed: {e}")
            return

        # --- Step 2: Move to side position of bottom drawer ---
        if bottom_side_pos is None:
            print("[Error] bottom_side_pos not found in positions.")
            return
        try:
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move to side pos!")
                return
        except Exception as e:
            print(f"[Error] Move to side pos failed: {e}")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        if bottom_anchor_pos is None:
            print("[Error] bottom_anchor_pos not found in positions.")
            return
        try:
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor pos!")
                return
        except Exception as e:
            print(f"[Error] Move to anchor pos failed: {e}")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # Use pick with approach from above (z axis)
        try:
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Pick drawer handle failed: {e}")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along x axis (assume drawer opens along +x)
        # Use a reasonable pull distance (e.g., 0.15m)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Pull failed: {e}")
            return

        # --- Step 6: Pick up the rubbish from the table ---
        if rubbish_pos is None:
            print("[Error] rubbish not found in positions.")
            return
        try:
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[Error] Pick rubbish failed: {e}")
            return

        # --- Step 7: Place the rubbish in the bin ---
        if bin_pos is None:
            print("[Error] bin not found in positions.")
            return
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Place rubbish failed: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()