import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.

        # --- Step 1: Rotate gripper to 90 degrees (vertical) for side approach ---
        # We'll use the bottom drawer as the "any available drawer"
        # Find the quaternion for 90 degrees rotation about z axis
        from scipy.spatial.transform import Rotation as R
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target: 90 deg about z axis from current orientation
            # For RLBench, 90 deg about z: [x, y, z, w] = [0, 0, sin(45°), cos(45°)]
            rot_90 = R.from_euler('z', 90, degrees=True)
            target_quat = rot_90.as_quat()  # xyzw
            # RLBench may use xyzw, so use as is
            print("[Task] Rotating gripper to 90 degrees about z axis.")
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print("[Task][Error] Exception during rotate:", e)
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            # Use 'bottom_side_pos' as the approach point for the bottom drawer
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            print("[Task] Moving to bottom drawer side position:", side_pos)
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print("[Task][Error] Exception during move to side position:", e)
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer (for grasping handle) ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            print("[Task] Moving to bottom drawer anchor position:", anchor_pos)
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print("[Task][Error] Exception during move to anchor position:", e)
            shutdown_environment(env)
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        try:
            # For picking the drawer handle, use the anchor position
            print("[Task] Grasping the bottom drawer handle at anchor position.")
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.05, approach_axis='z')
            if done:
                print("[Task] Task ended during pick (drawer handle)!")
                return
        except Exception as e:
            print("[Task][Error] Exception during pick (drawer handle):", e)
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open along the x axis ---
        try:
            # Pull distance: estimate (e.g., 0.15m)
            pull_distance = 0.15
            print("[Task] Pulling the bottom drawer open along x axis.")
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print("[Task][Error] Exception during pull:", e)
            shutdown_environment(env)
            return

        # --- Step 6: Pick up rubbish from the table ---
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise KeyError("rubbish not found in positions.")
            print("[Task] Picking up rubbish from table at:", rubbish_pos)
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick (rubbish)!")
                return
        except Exception as e:
            print("[Task][Error] Exception during pick (rubbish):", e)
            shutdown_environment(env)
            return

        # --- Step 7: Place rubbish in the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            print("[Task] Placing rubbish in bin at:", bin_pos)
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print("[Task][Error] Exception during place (rubbish in bin):", e)
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()