import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer to unlock the cabinet, then throw away the rubbish, leaving other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'rubbish', 'tomato1', 'tomato2', 'bin'

        # --- Step 1: Rotate gripper to 90 degrees (required for side approach) ---
        # We'll use the bottom drawer as the accessible one.
        # The gripper must be rotated to 'ninety_deg' (assume this is a quaternion for 90deg about z).
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Task] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to the anchor position of the bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Task] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # Use pick with approach_axis='y' (sideways) or 'z' (from above) as appropriate
        # We'll use 'y' as drawers are usually grasped from the front
        try:
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_axis='y')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the 'x' axis (assume drawer opens along +x)
        # We'll use a reasonable pull distance (e.g., 0.15m)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Task] Could not find 'rubbish' in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_axis='z')
            if done:
                print("[Task] Task ended during pick rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Task] Could not find 'bin' in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()