import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and likely mapping:
        # Drawers: 'bottom', 'middle', 'top'
        # Anchor/side/joint positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.
        # Items: 'item1', 'item2', 'item3', 'rubbish'
        # Bin: 'bin'
        # Gripper: assumed to be controlled via skills, not as a named object

        # For this plan, select a drawer (let's use 'bottom' drawer for determinism)
        drawer_name = 'bottom'
        anchor_pos_name = 'bottom_anchor_pos'
        side_pos_name = 'bottom_side_pos'
        # For the plan, we need the anchor and side positions for the drawer
        try:
            anchor_pos = positions[anchor_pos_name]
            side_pos = positions[side_pos_name]
        except KeyError:
            print(f"[Error] Could not find anchor or side position for drawer '{drawer_name}'.")
            shutdown_environment(env)
            return

        # For the 'move' actions, we need to move the gripper to the correct positions
        # For 'rotate', we need the quaternion for 90 degrees rotation (about z)
        # For 'pull', we need to pull along the x-axis (assume positive x opens the drawer)
        # For 'pick' and 'place', we need the positions of the rubbish and the bin

        # Find the rubbish object (from observation, it's 'rubbish')
        rubbish_name = 'rubbish'
        bin_name = 'bin'
        try:
            rubbish_pos = positions[rubbish_name]
            bin_pos = positions[bin_name]
        except KeyError:
            # If 'rubbish' is not in the object list, fallback to 'item3' as the likely rubbish
            print("[Warning] 'rubbish' not found in positions, using 'item3' as rubbish.")
            rubbish_name = 'item3'
            try:
                rubbish_pos = positions[rubbish_name]
            except KeyError:
                print("[Error] Could not find rubbish object position.")
                shutdown_environment(env)
                return
            try:
                bin_pos = positions[bin_name]
            except KeyError:
                print("[Error] Could not find bin object position.")
                shutdown_environment(env)
                return

        # Step 1: Move gripper to the side position of the bottom drawer
        print(f"[Step 1] Move to side position of drawer '{drawer_name}' at {side_pos}")
        obs, reward, done = move(
            env, task, target_pos=side_pos
        )
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # Step 2: Rotate gripper to 90 degrees (about z axis)
        # Get current orientation, then compute target quaternion for 90 deg about z
        print("[Step 2] Rotate gripper to 90 degrees about z axis")
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        from scipy.spatial.transform import Rotation as R
        # 90 deg about z axis
        rot_90 = R.from_euler('z', 90, degrees=True)
        target_quat = rot_90 * R.from_quat(current_quat)
        target_quat_xyzw = target_quat.as_quat()
        obs, reward, done = rotate(
            env, task, target_quat=target_quat_xyzw
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move gripper to anchor position of the drawer
        print(f"[Step 3] Move to anchor position of drawer '{drawer_name}' at {anchor_pos}")
        obs, reward, done = move(
            env, task, target_pos=anchor_pos
        )
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # Step 4: Pull the drawer open (assume pull along +x, distance 0.15m)
        print("[Step 4] Pull the drawer open along x axis")
        pull_distance = 0.15  # meters, adjust as needed for full open
        obs, reward, done = pull(
            env, task, pull_distance=pull_distance, pull_axis='x'
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 5: Move gripper above the rubbish
        print(f"[Step 5] Move to rubbish '{rubbish_name}' at {rubbish_pos}")
        obs, reward, done = move(
            env, task, target_pos=rubbish_pos
        )
        if done:
            print("[Task] Task ended after move to rubbish!")
            return

        # Step 6: Pick up the rubbish
        print(f"[Step 6] Pick up rubbish '{rubbish_name}' at {rubbish_pos}")
        obs, reward, done = pick(
            env, task, target_pos=rubbish_pos
        )
        if done:
            print("[Task] Task ended after pick!")
            return

        # Step 7: Move to bin
        print(f"[Step 7] Move to bin at {bin_pos}")
        obs, reward, done = move(
            env, task, target_pos=bin_pos
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 8: Place rubbish in bin
        print(f"[Step 8] Place rubbish '{rubbish_name}' in bin at {bin_pos}")
        obs, reward, done = place(
            env, task, target_pos=bin_pos
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()