import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock the cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object/Position Key Mapping ===
        # Map object names to keys in positions dict
        # For drawers, use anchor/side/joint positions as needed
        # For items and bin, use their keys directly
        # The following mapping assumes the object_positions module provides these keys

        # Drawer: open the bottom drawer (assumed accessible and not locked)
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        # Items
        item1_pos = positions.get('item1', None)
        item2_pos = positions.get('item2', None)
        item3_pos = positions.get('item3', None)
        # Bin
        bin_pos = positions.get('bin', None)

        # For robustness, check that all required positions are available
        required_keys = ['bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos', 'item1', 'bin']
        for key in required_keys:
            if positions.get(key, None) is None:
                print(f"[ERROR] Required object position '{key}' not found in positions dictionary.")
                shutdown_environment(env)
                return

        # === PLAN EXECUTION ===
        # Step 1: Move to the side of the bottom drawer
        print("[Step 1] Move to bottom_side_pos (side of bottom drawer)")
        [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        print("[Step 2] Rotate gripper to 90 degrees (drawer grasp orientation)")
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        [Frozen Code Start]
        obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after rotate to 90 deg!")
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasp)
        print("[Step 3] Move to bottom_anchor_pos (anchor of bottom drawer)")
        [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pull to open the drawer
        print("[Step 4] Pull to open the bottom drawer")
        pull_distance = 0.15
        [Frozen Code Start]
        obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after pull (open drawer)!")
            return

        # Step 5: Move to the trash item (item1)
        print("[Step 5] Move to item1 (trash)")
        [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=item1_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # Step 6: Pick the trash (item1)
        print("[Step 6] Pick item1 (trash)")
        [Frozen Code Start]
        obs, reward, done = pick(
                env,
                task,
                target_pos=item1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 7: Move to the bin
        print("[Step 7] Move to bin")
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 8: Place the trash in the bin
        print("[Step 8] Place item1 in bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()