import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names in the environment
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'item1', 'item2', 'item3', 'bin'
        # ]

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle grasp) ---
        # We'll assume the gripper's 90 degree orientation is provided by the environment
        # If not, we can construct it from the current pose
        try:
            # Get the current position and orientation
            obs = task.get_observation()
            # We'll rotate around the z-axis by 90 degrees
            from scipy.spatial.transform import Rotation as R
            current_quat = obs.gripper_pose[3:7]
            # Compose a 90 degree rotation about z
            rot_90 = R.from_euler('z', 90, degrees=True)
            current_rot = R.from_quat(current_quat)
            target_rot = rot_90 * current_rot
            target_quat = target_rot.as_quat()
        except Exception as e:
            print("[Error] Failed to compute 90 degree rotation:", e)
            shutdown_environment(env)
            return

        print("[Task] Rotating gripper to 90 degrees (z axis)")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (for grasping handle) ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer anchor position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # Use pick with approach from -z (downward)
        print("[Task] Grasping the bottom drawer handle at anchor position")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open (along x axis, e.g. 0.15m) ---
        print("[Task] Pulling the bottom drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # --- Step 6: Move gripper above the first rubbish item on the table (item1) ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        # Approach above item1 (add offset in z for safe approach)
        item1_above = np.array(item1_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above item1:", item1_above)
        obs, reward, done = move(env, task, target_pos=item1_above)
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # --- Step 7: Rotate gripper to 0 degrees (reset orientation for pick) ---
        # We'll rotate back to the original orientation (assume zero_deg is identity)
        try:
            # Get the current position and orientation
            obs = task.get_observation()
            # Identity quaternion (no rotation)
            target_quat_zero = np.array([0, 0, 0, 1])
        except Exception as e:
            print("[Error] Failed to compute zero degree rotation:", e)
            shutdown_environment(env)
            return

        print("[Task] Rotating gripper to 0 degrees (reset orientation)")
        obs, reward, done = rotate(env, task, target_quat=target_quat_zero)
        if done:
            print("[Task] Task ended after rotate to zero!")
            return

        # --- Step 8: Move to item1's pick position (directly above for pick) ---
        print("[Task] Moving to item1 pick position:", item1_pos)
        obs, reward, done = move(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # --- Step 9: Pick item1 from the table ---
        print("[Task] Picking item1 from the table")
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick (item1)!")
            return

        # --- Step 10: Move to bin position (for disposal) ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] Could not find 'bin' in object positions.")
            shutdown_environment(env)
            return

        # Approach above bin (add offset in z for safe drop)
        bin_above = np.array(bin_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above bin:", bin_above)
        obs, reward, done = move(env, task, target_pos=bin_above)
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # --- Step 11: Place item1 into the bin ---
        print("[Task] Placing item1 into the bin")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_axis='-z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

        # (Optional) Repeat for item2 and item3 if required by the goal
        # For brevity, only item1 is handled here as per the plan steps.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()