import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object/Position Key Mapping ---
        # We'll use the 'middle' drawer for this plan.
        # The relevant keys in positions:
        #   - 'middle_side_pos': side position for gripper to approach drawer
        #   - 'middle_anchor_pos': anchor position for gripping drawer handle
        #   - 'middle_joint_pos': position to pull to open drawer
        #   - 'item3': the rubbish object to pick and place
        #   - 'bin': the trash can

        # Defensive: check all required keys exist
        required_keys = [
            'middle_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'item3', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object/position '{k}' not found in positions dictionary.")

        # Step 1: Move to the side position of the middle drawer
        print("[Plan] Step 1: Move to middle_side_pos")
        obs, reward, done = move(
            env, task, target_pos=positions['middle_side_pos']
        )
        if done:
            print("[Task] Task ended after move to middle_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        from scipy.spatial.transform import Rotation as R
        quat_90_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(
            env, task, target_quat=quat_90_z
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position (drawer handle)
        print("[Plan] Step 3: Move to middle_anchor_pos")
        obs, reward, done = move(
            env, task, target_pos=positions['middle_anchor_pos']
        )
        if done:
            print("[Task] Task ended after move to anchor!")
            return

        # Step 4: Pick (grasp) the drawer handle (simulate pick-drawer)
        print("[Plan] Step 4: Pick (grasp) the drawer handle at anchor")
        obs, reward, done = pick(
            env, task, target_pos=positions['middle_anchor_pos'],
            approach_distance=0.10, approach_axis='z'
        )
        if done:
            print("[Task] Task ended after pick-drawer!")
            return

        # Step 5: Pull the drawer open (simulate pulling along x axis)
        print("[Plan] Step 5: Pull the drawer open")
        pull_vec = positions['middle_joint_pos'] - positions['middle_anchor_pos']
        pull_distance = np.linalg.norm(pull_vec)
        if pull_distance < 1e-3:
            raise RuntimeError("Pull distance too small; check anchor and joint positions.")
        pull_axis = 'x' if pull_vec[0] >= 0 else '-x'
        obs, reward, done = pull(
            env, task, pull_distance=pull_distance, pull_axis=pull_axis
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 6: Move gripper away from drawer (back to side position)
        print("[Plan] Step 6: Move to middle_side_pos (retreat)")
        obs, reward, done = move(
            env, task, target_pos=positions['middle_side_pos']
        )
        if done:
            print("[Task] Task ended after retreat move!")
            return

        # Step 7: Rotate gripper back to 0 degrees (for object pick)
        print("[Plan] Step 7: Rotate gripper to 0 degrees")
        quat_0_z = R.from_euler('z', 0, degrees=True).as_quat()
        obs, reward, done = rotate(
            env, task, target_quat=quat_0_z
        )
        if done:
            print("[Task] Task ended after rotate to 0!")
            return

        # Step 8: Move to the rubbish (item3) position
        print("[Plan] Step 8: Move to item3 (rubbish) position")
        obs, reward, done = move(
            env, task, target_pos=positions['item3']
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # Step 9: Pick up the rubbish (item3)
        print("[Plan] Step 9: Pick up item3 (rubbish)")
        obs, reward, done = pick(
            env, task, target_pos=positions['item3'],
            approach_distance=0.15, approach_axis='z'
        )
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # Step 10: Move to the bin
        print("[Plan] Step 10: Move to bin")
        obs, reward, done = move(
            env, task, target_pos=positions['bin']
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 11: Place the rubbish in the bin
        print("[Plan] Step 11: Place item3 in bin")
        obs, reward, done = place(
            env, task, target_pos=positions['bin'],
            approach_distance=0.15, approach_axis='z'
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[ERROR] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()