import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'bin'

        # === PLAN EXECUTION ===
        # Goal: Open one of the drawers in the cabinet, then put the rubbish in the bin.

        # We'll use the 'bottom' drawer for simplicity.
        # Plan:
        # 1. Rotate gripper to 90 degrees (vertical) for side approach.
        # 2. Move to the side position of the bottom drawer.
        # 3. Move to the anchor position of the bottom drawer.
        # 4. Pick the drawer handle (simulate by pick at anchor).
        # 5. Pull the drawer open.
        # 6. Pick the rubbish from the table.
        # 7. Place the rubbish in the bin.

        # Step 1: Rotate gripper to 90 degrees (vertical)
        try:
            print("[Step 1] Rotating gripper to 90 degrees (vertical)...")
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            # 90 deg about x axis: [0.7071068, 0, 0, 0.7071068]
            target_quat = np.array([0.7071068, 0, 0, 0.7071068])
            [Frozen Code Start]
    obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, side_pos)
obs, reward, done = move(env, task, anchor_pos)
obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='-z')
            [Frozen Code End]
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to the side position of the bottom drawer
        try:
            print("[Step 2] Moving to side position of bottom drawer...")
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            # [Frozen Code] move to side_pos
            if done:
                print("[Task] Task ended after move to side!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to anchor position of the bottom drawer
        try:
            print("[Step 3] Moving to anchor position of bottom drawer...")
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            # [Frozen Code] move to anchor_pos
            if done:
                print("[Task] Task ended after move to anchor!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick the drawer handle (simulate pick at anchor position)
        try:
            print("[Step 4] Picking the drawer handle (at anchor position)...")
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='-z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open
        try:
            print("[Step 5] Pulling the drawer open...")
            # Pull along x axis (assume drawer opens along +x), distance ~0.18m
            obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick the rubbish from the table
        try:
            print("[Step 6] Picking the rubbish from the table...")
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise KeyError("rubbish not found in positions.")
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick rubbish) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the rubbish in the bin
        try:
            print("[Step 7] Placing the rubbish in the bin...")
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 7 (place rubbish) failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()