import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock the cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'rubbish', 'tomato1', 'tomato2', 'bin'

        # --- Step 1: Rotate gripper to 90 degrees (for side approach to drawer) ---
        # We'll use the 'bottom' drawer as the accessible one (not locked in initial state)
        # The gripper starts at zero_deg; we want to rotate to ninety_deg
        # We'll use the quaternion for 90 deg rotation about z axis
        # Assume gripper's default orientation is aligned with world axes
        # We'll use scipy's Rotation to get the quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        target_euler = [0, 0, 90]  # degrees
        target_quat = R.from_euler('xyz', target_euler, degrees=True).as_quat()  # xyzw
        print("[Task] Rotating gripper to 90 degrees (side approach for drawer)")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # --- Step 2: Move to side position of bottom drawer ---
        # Move gripper to 'bottom_side_pos'
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Task] Error: 'bottom_side_pos' not found in object positions.")
            return
        print("[Task] Moving gripper to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended during move to bottom_side_pos!")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        # Move gripper to 'bottom_anchor_pos'
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Task] Error: 'bottom_anchor_pos' not found in object positions.")
            return
        print("[Task] Moving gripper to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Task ended during move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # Use pick skill to close gripper at anchor position
        print("[Task] Grasping (picking) the bottom drawer handle at anchor position")
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.05, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume drawer opens along +x)
        # Use a reasonable pull distance (e.g., 0.15m)
        pull_distance = 0.15
        print("[Task] Pulling the bottom drawer open along +x axis")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull (drawer)!")
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Task] Error: 'rubbish' not found in object positions.")
            return
        print("[Task] Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (rubbish)!")
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Task] Error: 'bin' not found in object positions.")
            return
        print("[Task] Placing rubbish in bin at:", bin_pos)
        obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()