# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and environment) ---
        # Drawer positions
        # For this plan, we will use the 'bottom' drawer (as any available drawer is acceptable)
        # The relevant positions for the bottom drawer:
        #   - 'bottom_side_pos' (side position for approach)
        #   - 'bottom_anchor_pos' (anchor position for grasping handle)
        #   - 'bottom_joint_pos' (pull direction reference)
        #   - 'waypoint1' (optional, for intermediate moves)
        #   - 'rubbish' (object to be picked and thrown away)
        #   - 'bin' (target location for rubbish)
        #
        # The plan is:
        # 1. Rotate gripper to 90 degrees (for side approach)
        # 2. Move to side position of bottom drawer
        # 3. Move to anchor position of bottom drawer
        # 4. Pick the drawer handle (simulate with pick at anchor)
        # 5. Pull the drawer open
        # 6. Pick up rubbish from table
        # 7. Place rubbish in bin

        # --- Step 1: Rotate gripper to 90 degrees (for side approach) ---
        # Get current gripper pose
        obs = task.get_observation()
        start_quat = obs.gripper_pose[3:7]
        # 90 degrees about z axis (assuming 'zero_deg' is default, 'ninety_deg' is 90 deg about z)
        # We'll use scipy to get the quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        target_rot = R.from_euler('z', 90, degrees=True)
        target_quat = target_rot.as_quat()  # xyzw
        # Ensure it's numpy array
        target_quat = np.array(target_quat)
        try:
            print("[Task] Step 1: Rotating gripper to 90 degrees about z axis.")
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
            print(f"[Task] Step 2: Moving to bottom_side_pos at {bottom_side_pos}")
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended after move to side!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
            print(f"[Task] Step 3: Moving to bottom_anchor_pos at {bottom_anchor_pos}")
            obs, reward, done = move(env, task, bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (simulate with pick at anchor position) ---
        # In this domain, 'pick-drawer' is not a primitive skill, so we use 'pick' at anchor position
        try:
            print(f"[Task] Step 4: Picking (grasping) at bottom_anchor_pos {bottom_anchor_pos}")
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along x axis (assume positive x is the pull direction for the drawer)
        # Use a reasonable pull distance (e.g., 0.15m)
        try:
            print("[Task] Step 5: Pulling the drawer open along x axis (0.15m)")
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up rubbish from table ---
        # Get rubbish position (assume it's on the table, as per initial state)
        try:
            rubbish_pos = positions['rubbish']
            print(f"[Task] Step 6: Picking up rubbish at {rubbish_pos}")
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place rubbish in bin ---
        try:
            bin_pos = positions['bin']
            print(f"[Task] Step 7: Placing rubbish in bin at {bin_pos}")
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
