import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For this environment, we expect keys like:
        # 'bottom_anchor_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_side_pos', 'top_anchor_pos', 'top_side_pos', 'rubbish', 'bin', etc.

        # --- Step 1: Rotate gripper to 90 degrees (vertical) for drawer manipulation ---
        # We'll use the bottom drawer as the "available" drawer (not locked, not open).
        # The gripper must be rotated to 90 degrees (vertical) to interact with the drawer handle.
        try:
            # Get current gripper pose
            obs = task.get_observation()
            # Target orientation: 90 degrees about x axis (RLBench uses xyzw)
            from scipy.spatial.transform import Rotation as R
            target_euler = [np.pi/2, 0, 0]  # 90deg about x
            target_quat = R.from_euler('xyz', target_euler).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, np.array(bottom_side_pos))
            if done:
                print("[Task] Task ended after move to side pos!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to bottom_side_pos: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to the anchor position of the bottom drawer (handle) ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, np.array(bottom_anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor pos!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to bottom_anchor_pos: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        try:
            # For picking the drawer handle, use the anchor position
            obs, reward, done = pick(env, task, np.array(bottom_anchor_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick on drawer handle: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open (along x axis, e.g., +x) ---
        try:
            # Pull distance: empirically, 0.15~0.20m is typical for a drawer
            pull_distance = 0.18
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise KeyError("rubbish not found in object positions.")
            obs, reward, done = pick(env, task, np.array(rubbish_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            obs, reward, done = place(env, task, np.array(bin_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()