# run_skeleton_task.py (Completed with Exploration Phase and Safety/Consistency Improvements)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For demonstration, print all available objects and their positions
        print("[Info] Object positions in environment:")
        for obj_id, pos in positions.items():
            print(f"  {obj_id}: {pos}")

        # === CONSOLIDATE OBJECT REPRESENTATION ===
        # Build a consistent object list
        object_ids = list(positions.keys())

        # === SAFETY CHECKS: Ensure all required objects exist ===
        required_objects = object_ids  # For exploration, we use all known objects
        for obj in required_objects:
            if obj not in positions:
                print(f"[Error] Required object '{obj}' not found in environment positions!")
                return

        # === STANDARDIZE GRIPPER STATE REPRESENTATION ===
        # We'll use only obs.gripper_pose for all gripper state checks
        def get_gripper_pose(obs):
            return getattr(obs, 'gripper_pose', None)

        # === FORCE CALIBRATION PLACEHOLDER ===
        # (Assume force calibration is handled in skill_code or by the environment)

        # === EXPLORATION PHASE: Try to discover missing predicates ===
        # The goal is to perform actions that would reveal missing predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known)
        # We'll attempt to move to each object's location, pick it up, and (if possible) pull it, logging any issues.

        robot_location = None
        # Try to infer robot's initial location from observation or environment
        try:
            robot_location = getattr(obs, 'robot_location', None)
            if robot_location is None:
                # Fallback: use a default or the first location in positions
                robot_location = list(positions.values())[0]
        except Exception as e:
            print(f"[Warning] Could not determine robot initial location: {e}")
            robot_location = list(positions.values())[0]

        # For each object, perform exploration actions
        for obj_id in object_ids:
            obj_pos = positions[obj_id]
            print(f"\n[Exploration] Investigating object '{obj_id}' at position {obj_pos}")

            # 1. MOVE: Move to the object's location (simulate 'identified' and 'temperature-known')
            try:
                print(f"[Action] Moving to object '{obj_id}' at {obj_pos}")
                obs, reward, done = move(
                    env,
                    task,
                    from_pos=robot_location,
                    to_pos=obj_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_location = obj_pos  # Update robot's current location
                print(f"[Result] Arrived at '{obj_id}' location.")
            except Exception as e:
                print(f"[Error] Failed to move to '{obj_id}': {e}")
                continue

            # 2. PICK: Attempt to pick the object (simulate 'weight-known' and 'durability-known')
            try:
                print(f"[Action] Picking up object '{obj_id}'")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Result] Picked up '{obj_id}'.")
            except Exception as e:
                print(f"[Error] Failed to pick '{obj_id}': {e}")
                continue

            # 3. PULL: If the object is a drawer or can be pulled, try to pull (simulate 'lock-known')
            # We'll check if the object name suggests it's a drawer or handle
            if 'drawer' in obj_id or 'handle' in obj_id:
                try:
                    print(f"[Action] Attempting to pull '{obj_id}'")
                    obs, reward, done = pull(
                        env,
                        task,
                        target_obj=obj_id,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Result] Pulled '{obj_id}'.")
                except Exception as e:
                    print(f"[Error] Failed to pull '{obj_id}': {e}")

            # 4. PLACE: Place the object back (if possible)
            try:
                print(f"[Action] Placing object '{obj_id}' back at {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Result] Placed '{obj_id}' back.")
            except Exception as e:
                print(f"[Error] Failed to place '{obj_id}': {e}")

            # Safety: If task ended, break
            if done:
                print(f"[Exploration] Task ended after interacting with '{obj_id}'.")
                break

        print("\n[Exploration] Completed exploration of all objects.")

        # === LOGGING AND DEBUGGING ===
        print("[Debug] Exploration phase complete. Check logs for missing predicate clues.")

    except Exception as main_e:
        print(f"[Fatal Error] Exception during skeleton task: {main_e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
