# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The exploration domain suggests that some knowledge predicates (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing in the main domain. We will attempt to use the available skills to explore and check for missing predicates.

        # For demonstration, let's try to "pull" on all objects that look like drawers or handles,
        # and see if we can discover a missing predicate (e.g., lock-known) by observing the environment's response.

        # We'll also attempt to "pick" and "move" to various objects/locations to see if any knowledge predicates are missing.

        # Get all object names and positions
        object_names = list(positions.keys())
        print("[Exploration] Objects in environment:", object_names)

        # Try to pick each object and see if any knowledge predicate is missing
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick object: {obj_name} at {obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not pick {obj_name}: {e}")

        # Try to pull on objects that might be drawers/handles
        for obj_name in object_names:
            if "drawer" in obj_name or "handle" in obj_name:
                obj_pos = positions[obj_name]
                print(f"[Exploration] Attempting to pull object: {obj_name} at {obj_pos}")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not pull {obj_name}: {e}")

        # Try to move to each object's position (if move skill is available and meaningful)
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to object: {obj_name} at {obj_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not move to {obj_name}: {e}")

        # Try to place any held object at a default location (if possible)
        # For demonstration, we use the first object's position as a place target
        if object_names:
            place_target = positions[object_names[0]]
            print(f"[Exploration] Attempting to place at {object_names[0]} position: {place_target}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_target,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing at {object_names[0]}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not place at {object_names[0]}: {e}")

        # Try to rotate the gripper to a default orientation (e.g., identity quaternion)
        print("[Exploration] Attempting to rotate gripper to default orientation")
        try:
            target_quat = np.array([0, 0, 0, 1])  # Identity quaternion
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Could not rotate gripper: {e}")

        print("[Exploration] Exploration phase complete. If any actions failed due to missing predicates, check logs above.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()