# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that some predicates (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing in the initial state or not being updated as expected.
        # We will perform an exploration sequence to try to discover which predicate is missing by using available skills.

        # For demonstration, let's assume we have a robot, a drawer, and an object (e.g., tomato) in the environment.
        # We'll try to move, pick, and pull, and observe if any predicate (e.g., lock-known) is not being updated.

        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']

        # We'll use try/except to handle missing objects or type mismatches gracefully.

        # Example object keys (replace with actual keys from your environment if needed)
        robot_name = 'robot'  # Placeholder, not used directly in skills
        drawer_name = None
        object_name = None
        drawer_handle_pos = None
        object_pos = None

        # Try to find a drawer and an object in the positions dictionary
        for key in positions:
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
                drawer_handle_pos = positions[key]
            elif ('tomato' in key or 'object' in key or 'ball' in key) and object_name is None:
                object_name = key
                object_pos = positions[key]

        # If not found, just pick the first two keys
        if drawer_name is None and len(positions) > 0:
            drawer_name = list(positions.keys())[0]
            drawer_handle_pos = positions[drawer_name]
        if object_name is None and len(positions) > 1:
            object_name = list(positions.keys())[1]
            object_pos = positions[object_name]

        print(f"[Exploration] Using drawer: {drawer_name}, object: {object_name}")

        # 1. Try to move to the drawer handle position (if move skill is available)
        #    In RLBench, move is often implicit in pick/place, but we call it for exploration.
        try:
            if drawer_handle_pos is not None:
                print(f"[Exploration] Moving to drawer handle at {drawer_handle_pos}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=drawer_handle_pos,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended during move to drawer handle!")
                    return
        except Exception as e:
            print(f"[Exploration] Move to drawer handle failed: {e}")

        # 2. Try to pick the object (e.g., tomato or ball)
        try:
            if object_pos is not None:
                print(f"[Exploration] Picking object {object_name} at {object_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended during pick!")
                    return
        except Exception as e:
            print(f"[Exploration] Pick failed: {e}")

        # 3. Try to pull the drawer (if pull skill is available)
        try:
            if drawer_handle_pos is not None:
                print(f"[Exploration] Pulling drawer {drawer_name}")
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=drawer_handle_pos,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended during pull!")
                    return
        except Exception as e:
            print(f"[Exploration] Pull failed: {e}")

        # 4. Try to rotate the gripper (if rotate skill is available)
        try:
            # For demonstration, rotate to a fixed quaternion (identity)
            print(f"[Exploration] Rotating gripper to identity quaternion")
            target_quat = np.array([0, 0, 0, 1])
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=50,
                threshold=0.05,
                timeout=5.0
            )
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Rotate failed: {e}")

        # 5. Try to place the object (if place skill is available)
        try:
            if object_pos is not None:
                # Place at a slightly offset position
                place_pos = np.array(object_pos) + np.array([0.05, 0, 0])
                print(f"[Exploration] Placing object {object_name} at {place_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended during place!")
                    return
        except Exception as e:
            print(f"[Exploration] Place failed: {e}")

        # === Exploration Complete ===
        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
