# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (is-open drawer1) is missing; we need to explore to determine if drawer1 is open.
        # We'll use available skills to check the state of drawer1.
        # Since we cannot directly check predicates, we attempt to pull the drawer and observe the result.

        drawer_name = 'drawer1'
        gripper_name = 'gripper'  # Assuming a single gripper; adjust if multiple
        # Positions for the drawer and gripper
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Error] '{drawer_name}' not found in object positions.")
            return

        # 1. Move gripper to the side position of the drawer (simulate approach)
        #    (Assume 'side' position is slightly offset from drawer front)
        side_offset = np.array([0.0, -0.15, 0.0])  # Example offset; adjust as needed
        side_pos = np.array(drawer_pos) + side_offset
        print(f"[Exploration] Moving gripper to side of {drawer_name} at {side_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to side.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to side: {e}")
            return

        # 2. Rotate gripper to align with drawer handle (simulate 'ninety_deg' rotation)
        #    (Assume 'ninety_deg' is a known angle; if not, skip or use default orientation)
        try:
            obs = task.get_observation()
            # If rotate skill requires angle names, use 'ninety_deg'
            obs, reward, done = rotate(env, task, gripper_name, 'zero_deg', 'ninety_deg')
            if done:
                print("[Exploration] Task ended unexpectedly during rotate.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")
            # If rotate is not needed or fails, continue

        # 3. Move gripper to anchor position (simulate grasping drawer handle)
        anchor_offset = np.array([0.0, -0.02, 0.0])  # Closer to handle
        anchor_pos = np.array(drawer_pos) + anchor_offset
        print(f"[Exploration] Moving gripper to anchor of {drawer_name} at {anchor_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to anchor: {e}")
            return

        # 4. Attempt to pick the drawer handle (simulate 'pick-drawer' action)
        #    Since 'pick-drawer' is not in available skills, use 'pick' if possible
        try:
            obs, reward, done = pick(env, task, drawer_name, None)  # Location is not needed for drawer
            if done:
                print("[Exploration] Task ended unexpectedly during pick.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            # If pick fails, continue

        # 5. Attempt to pull the drawer (this will reveal if it is openable)
        try:
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return
            print(f"[Exploration] Pulled {drawer_name}. If successful, (is-open {drawer_name}) is now true.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            # If pull fails, drawer may be locked or already open

        # 6. Exploration complete; now proceed with the main plan (if any)
        #    For this example, we only needed to determine (is-open drawer1)
        print("[Exploration] Exploration phase complete. Checked (is-open drawer1).")

        # === Main Task Plan (if any) ===
        # (Add further plan steps here if required by the task.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
