# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (is-open drawer0) ===
        # Feedback indicates that (is-open drawer0) is missing, so we need to explore and check the state of the drawer.
        # We'll use the available skills to interact with the drawer and determine if it is open.

        # Assumptions for object naming (adjust as needed for your environment):
        # - The drawer is named 'drawer0'
        # - The gripper is named 'gripper0'
        # - The anchor and side positions for the drawer are available in positions dict
        # - The angles are named 'zero_deg', 'ninety_deg' as in the PDDL

        drawer_name = 'drawer0'
        gripper_name = 'gripper0'
        anchor_pos_name = 'anchor_pos_drawer0'
        side_pos_name = 'side_pos_drawer0'
        angle_zero = 'zero_deg'
        angle_ninety = 'ninety_deg'

        # Get positions for the drawer manipulation
        try:
            drawer_pos = positions[drawer_name]
            gripper_pos = positions[gripper_name]
            anchor_pos = positions[anchor_pos_name]
            side_pos = positions[side_pos_name]
        except KeyError as e:
            print(f"[Error] Missing position for: {e}")
            shutdown_environment(env)
            return

        # 1. Move gripper to side position (if not already there)
        print("[Exploration] Moving gripper to side position of drawer.")
        try:
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Exploration] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # 2. Rotate gripper to 90 degrees (if not already)
        print("[Exploration] Rotating gripper to 90 degrees.")
        try:
            obs, reward, done = rotate(env, task, gripper_name, angle_zero, angle_ninety)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # 3. Move gripper to anchor position
        print("[Exploration] Moving gripper to anchor position of drawer.")
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # 4. Pick the drawer handle (pick-drawer is not in available skills, so we use pick if possible)
        # If pick-drawer is not available, skip this step or use pick if the handle is modeled as an object
        # For this code, we skip pick-drawer as it's not in available skills

        # 5. Pull the drawer to open it
        print("[Exploration] Pulling the drawer to check if it opens.")
        try:
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # 6. Check if the drawer is open (predicate: is-open drawer0)
        # This would typically be checked via the environment or observation
        # For demonstration, print the observation or state
        print("[Exploration] Checking if the drawer is open (is-open drawer0).")
        try:
            obs = task.get_observation()
            # The actual check depends on your observation structure
            # For example, if obs has a field 'drawer0_open' or similar:
            is_open = getattr(obs, 'drawer0_open', None)
            if is_open is not None:
                print(f"[Exploration] drawer0 open state: {is_open}")
            else:
                print("[Exploration] Could not determine drawer0 open state from observation.")
        except Exception as e:
            print(f"[Error] Exception during observation check: {e}")

        print("[Exploration] Exploration phase complete. Missing predicate (is-open drawer0) should now be known.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()