# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bottom_drawer': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate (is-open bottom) ---
        # Feedback indicates that the predicate (is-open bottom) is missing or needs to be explored.
        # We will attempt to open the 'bottom' drawer using available skills and observe the result.

        # Assumptions for object naming (adjust as needed based on your environment):
        # - Drawer name: 'bottom' (as in feedback)
        # - Gripper name: 'gripper' or similar
        # - Positions: positions['bottom'], positions['gripper'], etc.

        # Step 1: Move gripper to the side position of the bottom drawer
        try:
            gripper_pos = positions['gripper']
            bottom_side_pos = positions['bottom_side']  # Side position for bottom drawer
        except KeyError as e:
            print(f"[Error] Missing position for: {e}")
            shutdown_environment(env)
            return

        print("[Exploration] Moving gripper to side position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Exploration] Task ended during move to side position!")
            return

        # Step 2: Rotate gripper to 90 degrees (if required)
        # Assume angle names: 'zero_deg', 'ninety_deg'
        try:
            from_angle = 'zero_deg'
            to_angle = 'ninety_deg'
        except Exception as e:
            print(f"[Error] Angle names not defined: {e}")
            shutdown_environment(env)
            return

        print("[Exploration] Rotating gripper to 90 degrees.")
        try:
            obs, reward, done = rotate(env, task, gripper='gripper', from_angle=from_angle, to_angle=to_angle)
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Exploration] Task ended during rotate!")
            return

        # Step 3: Move gripper to anchor position of the bottom drawer
        try:
            bottom_anchor_pos = positions['bottom_anchor']
        except KeyError as e:
            print(f"[Error] Missing anchor position for: {e}")
            shutdown_environment(env)
            return

        print("[Exploration] Moving gripper to anchor position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Exploration] Task ended during move to anchor position!")
            return

        # Step 4: Pick the bottom drawer handle (simulate pick-drawer)
        # Since 'pick-drawer' is not in available skills, we use 'pick' if possible
        print("[Exploration] Attempting to pick the bottom drawer handle.")
        try:
            obs, reward, done = pick(env, task, obj='bottom_handle', loc='bottom_anchor')
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Exploration] Task ended during pick!")
            return

        # Step 5: Pull the bottom drawer to open it
        print("[Exploration] Pulling the bottom drawer to open it.")
        try:
            obs, reward, done = pull(env, task, obj='bottom', loc='bottom_anchor')
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Exploration] Task ended during pull!")
            return

        # Step 6: Check if the drawer is open (predicate is-open bottom)
        # This would be checked in the environment or via observation
        print("[Exploration] Checking if the bottom drawer is open (is-open bottom).")
        # (In a real system, you would check the observation or environment state here.)

        print("[Exploration] Exploration phase complete. Predicate (is-open bottom) should now be true if pull succeeded.")

        # --- End of Exploration Phase ---

        # (Continue with further plan steps as needed...)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
