# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bin1': (x1, y1, z1), 'object_1': (x2, y2, z2), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback: (at bin1 bin1) is present, which is likely an error or a missing predicate.
        # The exploration domain suggests that moving to a location allows the robot to "identify" objects at that location.
        # We will perform a move to bin1 to trigger any identification predicates and check the state.

        # For this example, we assume 'bin1' is a key in positions and is a location.
        # If not, handle gracefully.
        try:
            bin1_pos = positions['bin1']
        except KeyError:
            print("[Exploration] 'bin1' not found in object positions. Exploration aborted.")
            return

        print("[Exploration] Moving to bin1 to identify objects and check for missing predicates.")
        try:
            # The move skill may require only the environment, task, and target position.
            # If your move skill requires more arguments, adjust accordingly.
            obs, reward, done = move(
                env,
                task,
                target_pos=bin1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during move to bin1: {e}")
            return

        if done:
            print("[Exploration] Task ended during exploration move!")
            return

        # After moving, check the observation for any new predicates or information.
        obs = task.get_observation()
        print("[Exploration] Observation after move to bin1:", obs)

        # === Main Task Plan (Example) ===
        # If the missing predicate is related to identification, we can now proceed to pick/place as needed.
        # For demonstration, let's try to pick an object at bin1 if available.

        # Find an object at bin1 (excluding bin1 itself)
        object_to_pick = None
        for obj_name, obj_pos in positions.items():
            if obj_name != 'bin1':
                # For simplicity, assume objects within 0.1m of bin1 are "at" bin1
                if np.linalg.norm(np.array(obj_pos) - np.array(bin1_pos)) < 0.1:
                    object_to_pick = obj_name
                    break

        if object_to_pick is None:
            print("[Task] No object found at bin1 to pick. Task ends.")
            return

        print(f"[Task] Attempting to pick object '{object_to_pick}' at bin1.")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions[object_to_pick],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick of {object_to_pick}: {e}")
            return

        if done:
            print("[Task] Task ended after pick!")
            return

        # Place the object back at bin1 (or another location if desired)
        print(f"[Task] Placing object '{object_to_pick}' at bin1.")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=bin1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during place of {object_to_pick}: {e}")
            return

        if done:
            print("[Task] Task ended after place!")
            return

        print("[Task] Exploration and main plan completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()