# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Object List Verification ===
        # Ensure all objects referenced exist in the environment
        available_objects = set(positions.keys())
        print("[Info] Available objects in environment:", available_objects)

        # Example: Let's assume we expect 'rubbish', 'trash_can', 'drawer', 'gripper'
        expected_objects = ['rubbish', 'trash_can', 'drawer', 'gripper']
        for obj in expected_objects:
            if obj not in available_objects:
                print(f"[Warning] Expected object '{obj}' not found in environment. Skipping related actions.")

        # === Force Calibration Step (Exploration for Missing Predicate) ===
        # The feedback suggests we need to calibrate the gripper force before manipulation.
        # Since we do not have a direct skill for force calibration, we simulate this by
        # performing a gentle pick and release on a calibration object (if available).
        calibration_obj = None
        for obj in available_objects:
            if 'calibration' in obj or 'test' in obj:
                calibration_obj = obj
                break
        if calibration_obj:
            print(f"[Exploration] Calibrating gripper force using '{calibration_obj}'...")
            try:
                obs, reward, done = pick(env, task, target_pos=positions[calibration_obj])
                if done:
                    print("[Exploration] Task ended during calibration pick!")
                    return
                obs, reward, done = place(env, task, target_pos=positions[calibration_obj])
                if done:
                    print("[Exploration] Task ended during calibration place!")
                    return
            except Exception as e:
                print(f"[Exploration] Calibration failed: {e}")
        else:
            print("[Exploration] No calibration object found. Skipping force calibration.")

        # === Safety Check: Is Rubbish in Trash Can? ===
        # Only proceed if rubbish is in trash can before picking up
        rubbish_obj = 'rubbish' if 'rubbish' in available_objects else None
        trash_can_obj = 'trash_can' if 'trash_can' in available_objects else None
        if rubbish_obj and trash_can_obj:
            rubbish_pos = positions[rubbish_obj]
            trash_can_pos = positions[trash_can_obj]
            # Simple proximity check (could be improved with actual containment logic)
            if np.linalg.norm(np.array(rubbish_pos) - np.array(trash_can_pos)) > 0.2:
                print("[Safety] Rubbish is NOT in the trash can. Aborting pick action for safety.")
            else:
                print("[Safety] Rubbish is in the trash can. Proceeding with pick.")
                try:
                    obs, reward, done = pick(env, task, target_pos=rubbish_pos)
                    if done:
                        print("[Task] Task ended after picking rubbish!")
                        return
                except Exception as e:
                    print(f"[Error] Failed to pick rubbish: {e}")
        else:
            print("[Warning] Rubbish or trash can object not found. Skipping rubbish pick.")

        # === Example Drawer Manipulation (if present) ===
        drawer_obj = 'drawer' if 'drawer' in available_objects else None
        gripper_obj = 'gripper' if 'gripper' in available_objects else None
        if drawer_obj and gripper_obj:
            drawer_pos = positions[drawer_obj]
            gripper_pos = positions[gripper_obj]
            # Example: Rotate gripper, move to drawer, pull to open
            try:
                print("[Task] Rotating gripper to 90 degrees for drawer operation...")
                obs, reward, done = rotate(env, task, gripper=gripper_obj, from_angle='zero_deg', to_angle='ninety_deg')
                if done:
                    print("[Task] Task ended after gripper rotation!")
                    return
            except Exception as e:
                print(f"[Error] Failed to rotate gripper: {e}")
            try:
                print("[Task] Moving gripper to drawer position...")
                obs, reward, done = move(env, task, from_pos=gripper_pos, to_pos=drawer_pos)
                if done:
                    print("[Task] Task ended after moving to drawer!")
                    return
            except Exception as e:
                print(f"[Error] Failed to move to drawer: {e}")
            try:
                print("[Task] Pulling drawer open...")
                obs, reward, done = pull(env, task, gripper=gripper_obj, drawer=drawer_obj)
                if done:
                    print("[Task] Task ended after pulling drawer!")
                    return
            except Exception as e:
                print(f"[Error] Failed to pull drawer: {e}")
        else:
            print("[Info] Drawer or gripper not found. Skipping drawer manipulation.")

        # === Place Rubbish (if picked) ===
        if rubbish_obj and trash_can_obj:
            try:
                print("[Task] Placing rubbish into trash can...")
                obs, reward, done = place(env, task, target_pos=trash_can_pos)
                if done:
                    print("[Task] Task ended after placing rubbish!")
                    return
            except Exception as e:
                print(f"[Error] Failed to place rubbish: {e}")

        print("[Task] All planned actions completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()