# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'item3': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (gripper-at gripper approach_above_item3)
        # The missing predicate is that the gripper must be at 'approach_above_item3' before picking.
        # We need to ensure the gripper is at this position before attempting pick.

        # For demonstration, let's assume 'item3' is the object to pick,
        # and 'approach_above_item3' is a key in positions for the approach pose.
        # If not, adjust the key names as per your environment.

        object_to_pick = 'item3'
        approach_pose_key = 'approach_above_item3'
        pick_pose_key = 'item3'

        # Check if required keys exist in positions
        if approach_pose_key not in positions or pick_pose_key not in positions:
            print(f"[Error] Required positions '{approach_pose_key}' or '{pick_pose_key}' not found in object_positions.")
            return

        approach_pos = np.array(positions[approach_pose_key])
        pick_pos = np.array(positions[pick_pose_key])

        # === Plan Execution ===

        # 1. Move gripper to approach position above item3
        print(f"[Task] Moving gripper to approach position above {object_to_pick}: {approach_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=approach_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during approach move!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to approach: {e}")
            return

        # 2. Move gripper down to pick position (directly above item3)
        print(f"[Task] Moving gripper to pick position of {object_to_pick}: {pick_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=pick_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to pick position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to pick position: {e}")
            return

        # 3. Pick the object
        print(f"[Task] Picking up {object_to_pick} at {pick_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=pick_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            return

        # 4. (Optional) Place the object somewhere, if required by the plan
        # For demonstration, check if a 'place' location exists
        place_pose_key = 'place_location'
        if place_pose_key in positions:
            place_pos = np.array(positions[place_pose_key])
            print(f"[Task] Moving to place location: {place_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=place_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended during move to place location!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to place location: {e}")
                return

            print(f"[Task] Placing {object_to_pick} at {place_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Error] Exception during place: {e}")
                return

        # 5. (Optional) Additional actions (rotate, pull) can be added here if required by the plan

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()