# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper0': ..., 'drawer0': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (rotated gripper0 ninety_deg) ===
        # The feedback indicates that the predicate (rotated gripper0 ninety_deg) is missing in the initial state.
        # We need to ensure that the gripper is rotated to ninety_deg before performing actions that require this precondition.

        # Try to get the gripper and drawer positions from the environment
        try:
            gripper_name = 'gripper0'
            drawer_name = 'drawer0'
            # For demonstration, we assume these keys exist in positions
            gripper_pos = positions[gripper_name]
            drawer_pos = positions[drawer_name]
        except KeyError as e:
            print(f"[Error] Required object not found in positions: {e}")
            shutdown_environment(env)
            return

        # 1. Move the gripper to a safe position above the drawer (if needed)
        approach_height = 0.10  # 10cm above the drawer
        target_pos = np.array(drawer_pos)
        target_pos[2] += approach_height

        print(f"[Exploration] Moving gripper to above drawer at {target_pos}")
        try:
            obs, reward, done = move(env, task, target_pos)
            if done:
                print("[Exploration] Task ended during initial move!")
                return
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            shutdown_environment(env)
            return

        # 2. Rotate the gripper to ninety_deg (if not already)
        # We assume the angles are named 'zero_deg' and 'ninety_deg' in the environment
        # For demonstration, we use string names as in the PDDL
        from_angle = 'zero_deg'
        to_angle = 'ninety_deg'
        print(f"[Exploration] Rotating gripper {gripper_name} from {from_angle} to {to_angle}")
        try:
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # 3. (Optional) Confirm the rotated predicate is now true (if possible)
        # This would be done by checking the observation or state, if available

        print("[Exploration] Gripper rotated to ninety_deg. Exploration phase complete.")

        # === Main Task Plan ===
        # At this point, the missing predicate (rotated gripper0 ninety_deg) is established.
        # You can now proceed with the rest of the oracle plan using only the available skills.
        # For demonstration, we show a placeholder for the next steps.

        # Example: Pull the drawer open (if that's the next step in the plan)
        print(f"[Task] Attempting to pull drawer {drawer_name} with gripper {gripper_name}")
        try:
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Continue with further steps as required by the oracle plan, using only the available skills:
        # e.g., pick, place, move, rotate, pull, etc.

        print("[Task] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
