# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The goal is to explore the environment and use available skills to discover which predicate is missing.
        # We will attempt to use each skill in a safe, exploratory way and observe the effects.

        # 1. List all objects and locations from positions
        object_names = list(positions.keys())
        if not object_names:
            print("[Exploration] No objects found in the environment!")
            return

        # For demonstration, pick the first object as the exploration target
        target_obj = object_names[0]
        target_pos = positions[target_obj]

        print(f"[Exploration] Target object for exploration: {target_obj} at {target_pos}")

        # 2. Try to pick the object (if possible)
        try:
            print(f"[Exploration] Attempting to pick {target_obj}...")
            obs, reward, done = pick(
                env,
                task,
                target_pos=target_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Task ended after pick attempt on {target_obj}.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # 3. Try to place the object at a new location (if possible)
        # For exploration, try to place at a slightly offset position
        try:
            place_pos = (target_pos[0] + 0.1, target_pos[1], target_pos[2])
            print(f"[Exploration] Attempting to place {target_obj} at {place_pos}...")
            obs, reward, done = place(
                env,
                task,
                target_pos=place_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Task ended after place attempt on {target_obj}.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during place: {e}")

        # 4. Try to rotate the gripper (if possible)
        try:
            # For demonstration, rotate to a fixed quaternion (e.g., 90 degrees about z)
            from scipy.spatial.transform import Rotation as R
            quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            print(f"[Exploration] Attempting to rotate gripper to 90 deg about z axis...")
            obs, reward, done = rotate(
                env,
                task,
                target_quat=quat_90deg_z,
                max_steps=50,
                threshold=0.05,
                timeout=5.0
            )
            if done:
                print(f"[Exploration] Task ended after rotate attempt.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # 5. Try to pull (if possible)
        # For exploration, try to pull the object if the skill is available and the object is a drawer/handle
        try:
            print(f"[Exploration] Attempting to pull {target_obj} (if applicable)...")
            obs, reward, done = pull(
                env,
                task,
                target_obj=target_obj
            )
            if done:
                print(f"[Exploration] Task ended after pull attempt on {target_obj}.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # 6. Try to move (if possible)
        # If the move skill is available, attempt to move to another location (if locations are defined)
        try:
            # For demonstration, if there are at least two objects, use their positions as "locations"
            if len(object_names) > 1:
                from_obj = object_names[0]
                to_obj = object_names[1]
                from_pos = positions[from_obj]
                to_pos = positions[to_obj]
                print(f"[Exploration] Attempting to move from {from_obj} to {to_obj}...")
                obs, reward, done = move(
                    env,
                    task,
                    from_pos=from_pos,
                    to_pos=to_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after move attempt from {from_obj} to {to_obj}.")
                    return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")

        # 7. After exploration, print a summary
        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
