# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Align declared objects with actual objects ===
        # Feedback: Ensure all objects referenced are present in positions
        # We'll use only objects that exist in the environment
        available_objects = list(positions.keys())
        print("[Info] Available objects in environment:", available_objects)

        # Safety: Only interact with objects that exist in the environment
        def safe_get_pos(obj_name):
            if obj_name not in positions:
                print(f"[Warning] Object '{obj_name}' not found in environment. Skipping.")
                return None
            return positions[obj_name]

        # === Exploration Phase: Identify missing predicates ===
        # The exploration domain suggests that some predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known) may be missing.
        # Here, we perform exploration by moving to each location and attempting to pick/place/rotate/pull as appropriate,
        # and log any failures or missing information.

        # For demonstration, let's iterate through all objects and try to interact with them using available skills.
        # This will help us discover which predicates or object properties are missing.

        # We'll use only the provided skills: pick, place, move, rotate, pull

        # For each object, try to pick and place it (if possible), and try to rotate or pull if it's a drawer/handle.
        for obj_name in available_objects:
            obj_pos = safe_get_pos(obj_name)
            if obj_pos is None:
                continue  # Skip missing objects

            print(f"\n[Exploration] Attempting to interact with object: {obj_name}")

            # Try pick
            try:
                print(f"[Exploration] Trying to pick '{obj_name}' at {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Could not pick '{obj_name}': {e}")

            # Try place (if pick succeeded and hand is not empty)
            try:
                print(f"[Exploration] Trying to place '{obj_name}' at {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Could not place '{obj_name}': {e}")

            # Try rotate (if object is a gripper or has orientation)
            try:
                if 'gripper' in obj_name or 'drawer' in obj_name or 'handle' in obj_name:
                    print(f"[Exploration] Trying to rotate '{obj_name}'")
                    # For demonstration, use a fixed target quaternion (identity)
                    target_quat = np.array([0, 0, 0, 1])
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=50,
                        threshold=0.05,
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after rotating '{obj_name}'!")
                        break
            except Exception as e:
                print(f"[Exploration] Could not rotate '{obj_name}': {e}")

            # Try pull (if object is a drawer or handle)
            try:
                if 'drawer' in obj_name or 'handle' in obj_name:
                    print(f"[Exploration] Trying to pull '{obj_name}'")
                    obs, reward, done = pull(
                        env,
                        task,
                        obj_name
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling '{obj_name}'!")
                        break
            except Exception as e:
                print(f"[Exploration] Could not pull '{obj_name}': {e}")

        print("\n[Exploration] Completed interaction attempts with all available objects.")

        # === End of Exploration Phase ===

        # At this point, based on the feedback and exploration, you can analyze the logs to determine which predicates or object properties are missing.
        # For example, if pick fails due to missing 'identified' or 'weight-known', you can infer the missing predicate.

        # (Optional) You can add more sophisticated exploration or logging as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
