# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (is-side-pos) ===
        # Feedback: (is-side-pos side-pos-top top)
        # We need to check if the predicate (is-side-pos side-pos-top top) is true in the environment.
        # If not, we need to explore or log that this predicate is missing.

        # For demonstration, we will check if 'side-pos-top' and 'top' exist in positions,
        # and log the result. In a real scenario, this would involve more detailed state checking.

        side_pos_name = 'side-pos-top'
        drawer_name = 'top'

        print("[Exploration] Checking for missing predicate: (is-side-pos side-pos-top top)")
        if side_pos_name in positions and drawer_name in positions:
            print(f"[Exploration] Found positions for '{side_pos_name}' and '{drawer_name}'.")
            # In a real PDDL state, we would check if (is-side-pos side-pos-top top) holds.
            # Here, we simply log that the objects exist.
        else:
            print(f"[Exploration] Missing objects for predicate: (is-side-pos {side_pos_name} {drawer_name})")
            # Log or handle missing predicate as needed

        # === Example Plan Execution ===
        # Since the actual oracle plan is not provided, we demonstrate how to use the available skills.
        # You should replace this with the actual plan steps as needed.

        # Example: Move to the side position of the top drawer, rotate, and pull to open the drawer.

        try:
            # 1. Move gripper to the side position of the top drawer
            if side_pos_name in positions:
                target_pos = np.array(positions[side_pos_name])
                print(f"[Task] Moving to side position: {side_pos_name} at {target_pos}")
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Task] Task ended during move to side position!")
                    return
            else:
                print(f"[Task] Side position '{side_pos_name}' not found in positions.")
                return

            # 2. Rotate gripper to 90 degrees (assuming 'ninety_deg' is a known angle in the environment)
            # For demonstration, we assume the gripper is named 'gripper' and angles are handled in the environment.
            gripper_name = 'gripper'
            from_angle = 'zero_deg'
            to_angle = 'ninety_deg'
            print(f"[Task] Rotating gripper '{gripper_name}' from {from_angle} to {to_angle}")
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Task] Task ended during rotate!")
                return

            # 3. Move to anchor position (if defined)
            anchor_pos_name = 'anchor-pos-top'
            if anchor_pos_name in positions:
                anchor_pos = np.array(positions[anchor_pos_name])
                print(f"[Task] Moving to anchor position: {anchor_pos_name} at {anchor_pos}")
                obs, reward, done = move(env, task, anchor_pos)
                if done:
                    print("[Task] Task ended during move to anchor position!")
                    return
            else:
                print(f"[Task] Anchor position '{anchor_pos_name}' not found in positions.")
                return

            # 4. Pull the drawer open
            print(f"[Task] Pulling drawer '{drawer_name}' with gripper '{gripper_name}'")
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Task] Task ended during pull!")
                return

            # 5. (Optional) Pick and place an object if required by the plan
            # Example: pick an object from the drawer and place it somewhere
            object_to_pick = 'object_1'
            place_location = 'place_pos'
            if object_to_pick in positions and place_location in positions:
                pick_pos = np.array(positions[object_to_pick])
                place_pos = np.array(positions[place_location])
                print(f"[Task] Picking object '{object_to_pick}' at {pick_pos}")
                obs, reward, done = pick(env, task, object_to_pick, drawer_name)
                if done:
                    print("[Task] Task ended during pick!")
                    return
                print(f"[Task] Placing object '{object_to_pick}' at {place_pos}")
                obs, reward, done = place(env, task, object_to_pick, place_location)
                if done:
                    print("[Task] Task ended during place!")
                    return
            else:
                print(f"[Task] Either object '{object_to_pick}' or place location '{place_location}' not found in positions.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()