# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try all available skills to identify missing predicates ===
        # The goal is to determine which predicate is missing by attempting each skill and observing the result.
        # This is especially important if the planner times out or fails due to missing predicates.

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # For demonstration, we will attempt to execute each skill with plausible arguments.
        # If a skill fails due to a missing predicate, we log the error.
        # This helps identify which predicate is missing in the domain/problem.

        # We'll use the first available object and location for testing.
        # You may need to adjust these keys based on your environment.
        test_obj = None
        test_loc = None
        test_gripper = None
        test_drawer = None
        test_position = None
        test_angle_from = None
        test_angle_to = None

        # Try to extract plausible test arguments from positions and descriptions
        # (This is generic; adapt as needed for your environment)
        try:
            # Find an object and its position
            for k, v in positions.items():
                if 'drawer' in k:
                    test_drawer = k
                    test_position = v
                elif 'gripper' in k:
                    test_gripper = k
                elif 'angle' in k:
                    if test_angle_from is None:
                        test_angle_from = k
                    else:
                        test_angle_to = k
                elif 'pos' in k or 'position' in k:
                    test_position = v
                else:
                    test_obj = k
                    test_loc = v
            # Fallbacks if not found
            if test_obj is None and len(positions) > 0:
                test_obj = list(positions.keys())[0]
                test_loc = positions[test_obj]
        except Exception as e:
            print("[Exploration] Could not extract test arguments from positions:", e)

        print("=== [Exploration] Testing available skills to identify missing predicates ===")
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'pick':
                    # pick(env, task, target_pos, ...)
                    if test_obj is not None and test_loc is not None:
                        obs, reward, done = pick(
                            env,
                            task,
                            target_pos=test_loc,
                            approach_distance=0.15,
                            max_steps=50,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=5.0
                        )
                        exploration_results['pick'] = 'success'
                    else:
                        exploration_results['pick'] = 'skipped (no object/location)'
                elif skill == 'place':
                    # place(env, task, target_pos, ...)
                    if test_obj is not None and test_loc is not None:
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=test_loc,
                            approach_distance=0.15,
                            max_steps=50,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=5.0
                        )
                        exploration_results['place'] = 'success'
                    else:
                        exploration_results['place'] = 'skipped (no object/location)'
                elif skill == 'move':
                    # move(env, task, from_pos, to_pos, ...)
                    # We'll try to move from one position to another if possible
                    pos_keys = list(positions.keys())
                    if len(pos_keys) >= 2:
                        from_pos = positions[pos_keys[0]]
                        to_pos = positions[pos_keys[1]]
                        obs, reward, done = move(
                            env,
                            task,
                            from_pos=from_pos,
                            to_pos=to_pos,
                            approach_distance=0.15,
                            max_steps=50,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=5.0
                        )
                        exploration_results['move'] = 'success'
                    else:
                        exploration_results['move'] = 'skipped (not enough positions)'
                elif skill == 'rotate':
                    # rotate(env, task, gripper, from_angle, to_angle, ...)
                    if test_gripper is not None and test_angle_from is not None and test_angle_to is not None:
                        obs, reward, done = rotate(
                            env,
                            task,
                            gripper=test_gripper,
                            from_angle=test_angle_from,
                            to_angle=test_angle_to,
                            max_steps=50,
                            threshold=0.01,
                            timeout=5.0
                        )
                        exploration_results['rotate'] = 'success'
                    else:
                        exploration_results['rotate'] = 'skipped (missing gripper/angles)'
                elif skill == 'pull':
                    # pull(env, task, drawer, ...)
                    if test_drawer is not None:
                        obs, reward, done = pull(
                            env,
                            task,
                            drawer=test_drawer,
                            max_steps=50,
                            threshold=0.01,
                            timeout=5.0
                        )
                        exploration_results['pull'] = 'success'
                    else:
                        exploration_results['pull'] = 'skipped (no drawer)'
                else:
                    exploration_results[skill] = 'skipped (unknown skill)'
            except Exception as e:
                print(f"[Exploration] Skill '{skill}' failed with error: {e}")
                exploration_results[skill] = f'error: {e}'

        print("=== [Exploration] Results ===")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        # === End of Exploration Phase ===
        # At this point, you should have information about which skill(s) failed due to missing predicates.
        # Use this information to update your domain/problem PDDL as needed.

        # === (Optional) Main Task Plan Execution ===
        # If you have a valid plan, execute it step by step using the available skills.
        # For this template, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()