# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper1': ..., 'drawer1': ..., 'pos2': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (gripper-at gripper1 pos2)
        # The missing predicate is (gripper-at gripper1 pos2)
        # We need to ensure that the gripper is at pos2 before executing further actions.
        # This exploration phase will attempt to move the gripper to pos2 and check if the predicate holds.

        gripper_name = 'gripper1'
        target_position_name = 'pos2'
        try:
            gripper_pos = positions[gripper_name]
            pos2 = positions[target_position_name]
        except KeyError as e:
            print(f"[Exploration] Error: Missing object in positions: {e}")
            shutdown_environment(env)
            return

        print(f"[Exploration] Moving {gripper_name} to {target_position_name} at position {pos2}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(pos2))
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            shutdown_environment(env)
            return

        # After moving, check if the predicate (gripper-at gripper1 pos2) is satisfied
        # (Assume that the observation or environment state can be checked here if needed)
        print(f"[Exploration] Completed move to {target_position_name}.")
        # If further exploration is needed, add more checks here.

        # === Main Task Plan Execution ===
        # Now that the missing predicate is handled, proceed with the oracle plan.
        # Example: pick an object at pos2, place it somewhere, rotate, pull, etc.
        # Use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # For demonstration, let's assume we want to pick an object at pos2 and place it at another location.
        # Replace 'object1' and 'target_location' with actual keys from positions as needed.

        object_to_pick = None
        for key in positions:
            if key.startswith('object') or key.startswith('obj'):
                object_to_pick = key
                break
        if object_to_pick is None:
            print("[Task] No object found to pick at pos2.")
        else:
            print(f"[Task] Attempting to pick {object_to_pick} at {target_position_name}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(pos2),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
                shutdown_environment(env)
                return

            # Now place the object at a different location if available
            target_location = None
            for key in positions:
                if key != target_position_name and (key.startswith('pos') or key.startswith('location')):
                    target_location = key
                    break
            if target_location is not None:
                print(f"[Task] Placing {object_to_pick} at {target_location}")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(positions[target_location]),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Task] Task ended after place!")
                        return
                except Exception as e:
                    print(f"[Task] Exception during place: {e}")
                    shutdown_environment(env)
                    return

        # Example: Rotate and pull actions if required by the plan
        # These require specific parameters; adjust as per your environment and plan
        # For demonstration, we check if 'drawer1' and angle names exist
        if 'drawer1' in positions and 'angle_from' in positions and 'angle_to' in positions:
            try:
                print("[Task] Rotating gripper1 from angle_from to angle_to")
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name='gripper1',
                    from_angle='angle_from',
                    to_angle='angle_to'
                )
                if done:
                    print("[Task] Task ended after rotate!")
                    return
            except Exception as e:
                print(f"[Task] Exception during rotate: {e}")
                shutdown_environment(env)
                return

            try:
                print("[Task] Pulling drawer1 with gripper1")
                obs, reward, done = pull(
                    env,
                    task,
                    gripper_name='gripper1',
                    drawer_name='drawer1'
                )
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pull: {e}")
                shutdown_environment(env)
                return

        print("[Task] Main plan execution completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()