# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer': ..., 'anchor-pos-bottom': ..., 'bottom': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (anchor-pos anchor-pos-bottom bottom)
        # The missing predicate is likely (is-anchor-pos anchor-pos-bottom bottom drawer)
        # We need to explore and check if the anchor position for the drawer is correctly set.

        # Try to find anchor positions for the drawer
        # We'll print out all positions and look for anchor/side positions
        print("[Exploration] Available object positions:")
        for name, pos in positions.items():
            print(f"    {name}: {pos}")

        # Try to find anchor position candidates
        anchor_pos_name = None
        drawer_name = None
        for name in positions:
            if "anchor" in name or "anchor-pos" in name:
                anchor_pos_name = name
            if "drawer" in name:
                drawer_name = name

        if anchor_pos_name is None:
            print("[Exploration] No anchor position found in object positions! Exploration failed.")
        else:
            print(f"[Exploration] Anchor position candidate: {anchor_pos_name}")

        if drawer_name is None:
            print("[Exploration] No drawer object found in object positions! Exploration failed.")
        else:
            print(f"[Exploration] Drawer object candidate: {drawer_name}")

        # For demonstration, let's assume we want to move the gripper to the anchor position and try to pick the drawer
        # This will help us verify if the predicate (is-anchor-pos ...) is missing or not

        # === Task Plan Execution ===
        # 1. Move to anchor position
        # 2. Rotate gripper if needed (assume 'ninety_deg' is required for side/anchor)
        # 3. Pick the drawer at anchor position
        # 4. Pull the drawer

        # Step 1: Move to anchor position
        if anchor_pos_name is not None:
            anchor_pos = positions[anchor_pos_name]
            print(f"[Task] Moving to anchor position: {anchor_pos_name} at {anchor_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
                if done:
                    print("[Task] Task ended during move to anchor position!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to anchor position: {e}")
                return
        else:
            print("[Task] Cannot proceed: anchor position not found.")
            return

        # Step 2: Rotate gripper to required angle (if needed)
        # For this example, we assume the gripper must be rotated to 'ninety_deg'
        # We need to get the current angle and rotate if not already at 'ninety_deg'
        # Since we don't have direct access to angle names, we skip this if not available

        # Step 3: Pick the drawer at anchor position
        # We need the drawer name and anchor position name
        if drawer_name is not None and anchor_pos_name is not None:
            print(f"[Task] Attempting to pick the drawer '{drawer_name}' at anchor position '{anchor_pos_name}'")
            try:
                # The pick skill is for objects, but for drawers, the domain uses 'pick-drawer'
                # However, only 'pick' is available in the provided skills, so we use 'pick'
                obs, reward, done = pick(env, task, obj=drawer_name, loc=anchor_pos_name)
                if done:
                    print("[Task] Task ended during pick of drawer!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pick of drawer: {e}")
                return
        else:
            print("[Task] Cannot proceed: drawer or anchor position not found.")
            return

        # Step 4: Pull the drawer
        print(f"[Task] Attempting to pull the drawer '{drawer_name}'")
        try:
            obs, reward, done = pull(env, task, obj=drawer_name)
            if done:
                print("[Task] Task ended during pull of drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull of drawer: {e}")
            return

        print("[Task] Drawer should now be open if all actions succeeded.")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()